#include <string.h>
#include <memory.h>
#include <malloc.h>
#include <limits.h>

#include "util.h"


/**************************************************************** Prototypes */

/* ODOT...
 */
static OGNode *
       ogCreateNode(OGraph * pog, void *pe);

/* ODOT...
 */
static void
     ogFreeNode(OGNode * pgn);

/* ODOT...
 */
static void
     ogAddSucc(OGNode * pgna, OGNode * pgnb);

/* ODOT...
 */
static void *
     ogKeyFunc(OGNode * pgn);

/* ODOT...
 */
static void
     ogPrintFunc(OGNode * pgn);

/* ODOT...
 */
static void
     ogSetFlags(OGraph * pog, int flag);

/* ODOT...
 */
static void
     ogFindTreeRootAux(OGNode * pogn, Tree * pt, TreeNode * ptnf);

/* ODOT...
 */
static Tree *
     ogFindTreeRoot(OGNode * pogn);

/******************************************************************** Bodies */





/*****************************************************************************\
*                                                                             *
* Oriented Graph                                                              *
*                                                                             *
\*****************************************************************************/

/* ODOT...
 */
static OGNode *
ogCreateNode(OGraph * pog, void *pe)
{
    OGNode *pgn;

    pgn = (OGNode *) UMallocMsg(sizeof(OGNode), "new node for Oriented Graph");
    pgn->pe = pe;
    pgn->flag = 0;
    pgn->nbs = 0;
    pgn->nbp = 0;
    pgn->sa = NULL;
    pgn->pog = pog;
    return pgn;
}

/* ODOT...
 */
static void
ogFreeNode(OGNode * pgn)
{
    (*(pgn->pog->freeFunc)) (pgn->pe);
    free(pgn);
}

/* ODOT...
 */
static void
ogAddSucc(OGNode * pgna, OGNode * pgnb)
{
    if ((pgna->nbs % OGNSUCC_CHKSZ) == 0) {
	if (pgna->sa == NULL)
	    pgna->sa = (OGNode **) UMalloc(OGNSUCC_CHKSZ * sizeof(OGNode *));
	else
	    pgna->sa = (OGNode **) URealloc(pgna->sa, (pgna->nbs + OGNSUCC_CHKSZ) * sizeof(OGNode *));
    }
    pgna->sa[pgna->nbs] = pgnb;
    pgnb->nbp++;
    pgna->nbs++;
}

/* ODOT...
 */
static void *
ogKeyFunc(OGNode * pgn)
{
    return (*(pgn->pog->keyFunc)) (pgn->pe);
}

/* ODOT...
 */
static void
ogPrintFunc(OGNode * pgn)
{
    (*(pgn->pog->printFunc)) (pgn->pe);
}


/* ODOT...
 */
OGraph *
       ogCreate(int orderOfSize, void *(*keyFunc) (), int (*orderFunc) (), void (*freeFunc) (), int (*hashFunc) (),
		void (*printFunc) ())
{
    OGraph *pg;

    if ((keyFunc == NULL) || (orderFunc == NULL) || (freeFunc == NULL) || (hashFunc == NULL))
	UIError("createOGraph", "bad parameters");
    pg = (OGraph *) UMallocMsg(sizeof(OGraph), "createOGraph: ");
    pg->es = orderOfSize;
    pg->nbn = 0;
    pg->keyFunc = keyFunc;
    pg->ordFunc = orderFunc;
    pg->freeFunc = freeFunc;
    pg->hashFunc = hashFunc;
    if (printFunc == NULL)
	printFunc = printVoid;
    pg->printFunc = printFunc;
    pg->ht = htCreate(pg->es / 10 + 1, ogKeyFunc, orderFunc, ogFreeNode, hashFunc, ogPrintFunc);
    return pg;
}

/* ODOT...
 */
void
ogFree(OGraph * pog)
{
    OGNode *pogn;
    int i;

    for (i = 0; i < pog->nbn; i++) {
	pogn = htLookNth(pog->ht, i);
	free(pogn->sa);
    }
    htFree(pog->ht, True);
    free(pog);
}

/* ODOT...
 */
void
ogAddNode(OGraph * pog, void *pe)
{
    OGNode tmp;

    tmp.pog = pog;			/* simulate a nod for compatibility
					 * with ogKeyFunc */
    tmp.pe = pe;
    if (!(htSearch(pog->ht, &tmp))) {
	htInsert(pog->ht, (void *) ogCreateNode(pog, pe), True);
	pog->nbn++;
    }
}


/* Add an arc between node pa and pb (pa->pb).
 * Nodes are created if needed.
 */
void
ogAddArc(OGraph * pog, void *pa, void *pb)
{
    OGNode *pgna, *pgnb;
    OGNode tmp;

    tmp.pog = pog;
    tmp.pe = pa;
    if (!(pgna = htSearch(pog->ht, &tmp))) {
	pgna = ogCreateNode(pog, pa);
	htInsert(pog->ht, (void *) pgna, True);
	pog->nbn++;
    }
    tmp.pe = pb;
    if (!(pgnb = htSearch(pog->ht, &tmp))) {
	pgnb = ogCreateNode(pog, pb);
	htInsert(pog->ht, (void *) pgnb, True);
	pog->nbn++;
    }
    ogAddSucc(pgna, pgnb);
}

/* ODOT...
 */
static void
ogSetFlags(OGraph * pog, int flag)
{
    int i;
    OGNode *pogn;

    for (i = 1; i <= pog->nbn; i++) {
	pogn = (OGNode *) htLookNth(pog->ht, i);
	pogn->flag = flag;
    }
}

/* ODOT...
 */
static void
ogFindTreeRootAux(OGNode * pogn, Tree * pt, TreeNode * ptnf)
{
    TreeNode *ptn;
    int i;

    if (pogn->flag == 0) {
	ptn = tAddNode(pt, ptnf, pogn->pe);
	pogn->flag = 1;
	for (i = 0; i < pogn->nbs; i++) {
	    ogFindTreeRootAux(pogn->sa[i], pt, ptn);
	}
    }
}

/* ODOT...
 */
static Tree *
ogFindTreeRoot(OGNode * pogn)
{
    Tree *pt;

    pt = tCreate(pogn->pog->keyFunc, pogn->pog->ordFunc, pogn->pog->freeFunc, pogn->pog->printFunc);
    ogSetFlags(pogn->pog, 0);
    ogFindTreeRootAux(pogn, pt, (TreeNode *) NULL);
    return pt;
}

/* ODOT...
 */
Bools
ogCheckCycleAux(OGNode * pogn)
{
    int i;

    if (pogn->flag == 2)
	return True;			/* this node is marked, so it's is
					 * own ancester. cycle! */
    if (pogn->flag == 1)
	return False;			/* this is just an old one. it's
					 * already checked */
    pogn->flag = 2;			/* now it's an ancester */
    for (i = 0; i < pogn->nbs; i++) {	/* let's check these sons */
	if (ogCheckCycleAux(pogn->sa[i]) == True)
	    return True;
    }
    pogn->flag = 1;			/* all this branch has been checked */
    return False;
}

/* Does this oriented graph embed a cycle ?
 */
Bools
ogCheckCycle(OGraph * pog)
{
    int i;
    OGNode *pogn;

    for (i = 1; i <= pog->nbn; i++) {
	pogn = htLookNth(pog->ht, i);
	if (pogn->flag == 1)
	    continue;			/* this node has already been checked */
	if (ogCheckCycleAux(pogn) == True)
	    return True;
    }
    return False;
}


/* Find an order in an oriented graph.
 *
 * Return: the returned list is ordered from most prior to less prior
 * in the graph.
 *
 * Example:
 *   graph:   a-> b ->c     list:  a->b->c->d  or  a->d->b->c
 * \> d
 *
 * Warning: graph must not be cyclic
 */
List *
ogFindOrder(OGraph * pog)
{
    int i, j, k;
    List *pl;
    OGNode **aogn;

    pl = lCreate(pog->keyFunc, pog->ordFunc, pog->freeFunc, pog->printFunc);
    aogn = (OGNode **) UMalloc(pog->nbn * sizeof(OGNode *));
    for (i = 0; i < pog->nbn; i++) {
	aogn[i] = htLookNth(pog->ht, i);/* build a fast access struct for
					 * nodes */
	aogn[i]->flag = aogn[i]->nbp;
    }
    for (i = 0; i < pog->nbn; i++) {	/* to order all the nodes */
	for (j = 0; j < pog->nbn - i; j++) {	/* the unordered nodes are in
						 * the head of aogn */
	    if (aogn[j]->flag == 0) {	/* this node has no predecessor */
		lAdd(pl, aogn[j]->pe);	/* take this node. it's the next in
					 * this order */
		for (k = 0; k < aogn[j]->nbs; k++)	/* cut it off the graph */
		    aogn[j]->sa[k]->flag--;	/* it's no longer a
						 * predecessor */
		aogn[j] = aogn[(pog->nbn) - i - 1];
		/* to keep the unordered nodes in the head */
		break;
	    }
	}
    }
    free(aogn);
    lReverse(pl);
    return pl;
}

/* ODOT...
 */
Tree *
ogFindTree(OGraph * pog)
{
    int i;
    Tree *pt;

    for (i = 1; i <= pog->nbn; i++) {
	pt = ogFindTreeRoot((OGNode *) htLookNth(pog->ht, i));
	if (pt->nbn == pog->nbn)
	    return pt;
	else
	    tFree(pt, False);
    }
    return (Tree *) NULL;
}
