#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>

#include "util.h"


#define INDENT_TAB 3

/* used by UIntBigEndianEnc(Dec) */
static char *UIntByteOrder;

/**************************************************************** Prototypes */

/* Fill byteOrder.
 *
 * Using byteOrder functions like UIntBigEndianEnc(Dec) are able to en(de)code.
 * int independantly of host architecture
 */
static void
     UFillByteOrder(char *byteOrder);

/* ODOT...
 */
static void
     UInitIntByteOrder(void);

/******************************************************************** Bodies */


/* Like std select but restart if interrupted
 */
int
URestartSelect(int width, fd_set * readfds, fd_set * writefds, fd_set * exceptfds, struct timeval * timeout)
{
    int status;

    while (((status = select(width, readfds, writefds, exceptfds, timeout)) < 0) && (errno == EINTR));	/* empty */
    return status;
}


/* There is no standard library function that you can count on in
 * all environments for "napping" (the usual name for short
 * sleeps).  Some environments supply a "usleep(n)" function which
 * suspends execution for n microseconds.  If your environment
 * doesn't support usleep(), here are a couple of implementations
 * for BSD and System V environments.
 *
 * History: The following code is adapted from Doug Gwyn's System V emulation
 * support for 4BSD and exploits the 4BSD select() system call.
 * Doug originally called it 'nap()'; you probably want to call it
 * "usleep()";
 *
 * Returns:  0 if ok, else -1
 */
int
UUsleep(long usec /* Delay in microseconds */ )
{
    struct timeval delay;

    delay.tv_sec = 0L;
    delay.tv_usec = (long) (1000 * usec);
    return URestartSelect(0, (fd_set *) 0, (fd_set *) 0, (fd_set *) 0, &delay);
}

/* Fill byteOrder.
 *
 * Using byteOrder functions like UIntBigEndianEnc(Dec) are able to en(de)code.
 * int independantly of host architecture
 */
static void
UFillByteOrder(char *byteOrder)
{
    char *p;
    int test, i, j;

    p = (char *) &test;
    test = 1;
    for (i = 0; i < sizeof(int); i++) {
	for (j = 0; j < sizeof(int); j++) {
	    if (*(p + j)) {
		byteOrder[i] = j;
		break;
	    }
	}
	test *= 256;
    }
}

/* ODOT...
 */
static void
UInitIntByteOrder(void)
{
    UIntByteOrder = (char *) UMalloc(sizeof(char) * sizeof(int));

    UFillByteOrder(UIntByteOrder);
}

/* ODOT...
 */
int
UIntBigEndianEnc(int toEnc)
{
    int enc, i;
    char *pe, *pte;

    if (!UIntByteOrder)
	UInitIntByteOrder();
    pe = (char *) &enc;
    pte = (char *) &toEnc;
    for (i = 0; i < sizeof(int); i++) {
	*(pe++) = pte[UIntByteOrder[i]];
    }
    return enc;
}

/* ODOT...
 */
int
UIntBigEndianDec(int toDec)
{
    int dec, i;
    char *pd, *ptd;

    if (!UIntByteOrder)
	UInitIntByteOrder();
    pd = (char *) &dec;
    ptd = (char *) &toDec;
    for (i = 0; i < sizeof(int); i++) {
	*(pd++) = ptd[UIntByteOrder[i]];
    }
    return dec;
}

/* Convert a byte into a string reflecting it's bit pattern.
 * For other types just change char into the desired type
 * (don't work with types with size > sizeof(long)).
 */
char *
UCharToBit(char w)
{
    unsigned long mask = 1;
    unsigned long tw = (unsigned long) w;
    char *rep;
    char *pt, c;
    int len, pak, i;

    len = sizeof(char) * 8;		/* just change char in the desired
					 * type */

    len += len / 4 - 1;
    rep = (char *) malloc(len + 1);
    rep[len] = '\0';
    pt = rep + len;
    pak = 0;
    for (i = 0; i < sizeof(int) * 8; i++) {
	c = (tw & mask) ? '1' : '0';
	mask <<= 1;
	*pt = c;
	pt--;
	pak++;
	if (pak == 4) {
	    pak = 0;
	    *pt = ' ';
	    pt--;
	}
    }
    while (pt >= rep) {
	*pt = ' ';
	pt--;
    }
    return rep;
}

/* Convert an int into a string reflecting it's bit pattern
 */
char *
UIntToBit(int w)
{
    unsigned long mask = 1;
    unsigned long tw = (unsigned long) w;
    char *rep;
    char *pt, c;
    int len, pak, i;

    len = sizeof(int) * 8;

    len += len / 4 - 1;
    rep = (char *) malloc(len + 1);
    rep[len] = '\0';
    pt = rep + len;
    pak = 0;
    for (i = 0; i < sizeof(int) * 8; i++) {
	c = (tw & mask) ? '1' : '0';
	mask <<= 1;
	*pt = c;
	pt--;
	pak++;
	if (pak == 4) {
	    pak = 0;
	    *pt = ' ';
	    pt--;
	}
    }
    while (pt >= rep) {
	*pt = ' ';
	pt--;
    }
    return rep;
}


/* Check if file named 'n' exist with mod 'p'
 */
Bools
UFileExist(char *n,
	   char *p			/* is a string containing:x->
					 * execute, r-> read, w-> write, d->
	       directory */ )
{
    int status;
    struct stat buf;

    status = stat(n, &buf);

    if (strchr(p, 'x')) {
	if (!(buf.st_mode & S_IXUSR))
	    return False;
    }
    if (strchr(p, 'r')) {
	if (!(buf.st_mode & S_IRUSR))
	    return False;
    }
    if (strchr(p, 'w')) {
	if (!(buf.st_mode & S_IWUSR))
	    return False;
    }
    if (strchr(p, 'd')) {
	if (!S_ISDIR(buf.st_mode))
	    return False;
    }
    return True;
}


/* Dump memory.
 *
 * Return: a string (caller must free it after use)
 */
char *
UDumpMem(char *pMem, int len, char mod)
{
    int i;
    char *pTailDump, *pDump;

    switch (mod) {
    case UDUMP_HEX:{
	    pDump = (char *) UMalloc(sizeof(char) * len * 3 + 1);

	    pTailDump = pDump;
	    for (i = 0; i < len; i++) {
		sprintf(pTailDump, "%X ", *pMem++);
		pTailDump += 3;
	    }
	    break;
	}
    case UDUMP_ASC:{
	    pDump = (char *) UMalloc(sizeof(char) * len + 1);

	    pTailDump = pDump;
	    for (i = 0; i < len; i++) {
		if (isprint(*pMem))
		    *pTailDump = *pMem;
		else
		    *pTailDump = '.';
		pTailDump++;
		pMem++;
	    }
	    break;
	}
    case UDUMP_ASC_OR_DEC:{
	    int rLen = 0;

	    for (i = 0; i < len; i++) {
		if (isprint(pMem[i]))
		    rLen++;
		else
		    rLen += 4;
	    }
	    pDump = (char *) UMalloc(sizeof(char) * rLen + 1);

	    pTailDump = pDump;
	    for (i = 0; i < len; i++) {
		if (isprint(*pMem)) {
		    *pTailDump = *pMem;
		    pTailDump++;
		}
		else {
		    sprintf(pTailDump, "[%2d]", *pMem);
		    pTailDump += 4;
		}
		pMem++;
	    }
	    break;
	}
    case UDUMP_RAW:{
	    pDump = (char *) UMalloc(sizeof(char) * len + 1);

	    pTailDump = pDump;
	    bcopy(pTailDump, pMem, len);
	    pTailDump += len;
	    break;
	}
    default:{
	    UIError("UDumpMem", "unknown mode %d", mod);
	}
    }
    *pTailDump = '\0';
    return pDump;
}


/* Create (start) a new chrono.
 *
 * It is set to zero.
 *
 * Return: chrono
 */
Chrono *
UChronoCreate(void)
{
    Chrono *pc;

    pc = UNew(Chrono);
    gettimeofday(&pc->start, NULL);
    return pc;
}

/* Restart chrono.
 *
 * Reset to zero.
 */
void
UChronoRestart(Chrono * pc)
{
    gettimeofday(&pc->start, NULL);
}

/* Get elapsed time since last start (or restart).
 *
 * Return: number of milliseconds elapsed.
 *
 * Warning: be aware of overfull (time may exceed capacity of a long)
 */
long
UChronoGet(Chrono * pc)
{
    struct timeval now;
    long dt;

    gettimeofday(&now, NULL);
    dt = 1000 * (now.tv_sec - (pc->start).tv_sec) + (now.tv_usec - (pc->start).tv_usec);
    return dt;
}


void
UIndent(char *com)
{
    static char *spaces = NULL;
    static int nbSpaces = 0;
    static int level;
    char *c, *comb;
    int i;


    if (com == (char *) NULL)
	com = "p";
    comb = UStrDup(com);
    while ((c = strchr(comb, '+')) != (char *) NULL) {
	*c = ' ';
	level++;
    }
    while ((c = strchr(comb, '-')) != (char *) NULL) {
	*c = ' ';
	level--;
    }
    if (level < 0)
	UIError("agat server", "%s", "indent: level <0!");
    if ((c = strchr(comb, 'p')) != (char *) NULL) {
	*c = ' ';
	if ((nbSpaces < level * INDENT_TAB) || (spaces == (char *) NULL)) {
	    nbSpaces = level * INDENT_TAB + 4 * INDENT_TAB;
	    if (spaces != (char *) NULL)
		free(spaces);
	    spaces = (char *) UMalloc(nbSpaces + 1);
	    for (i = 0; i < nbSpaces; i++)
		spaces[i] = ' ';
	    spaces[nbSpaces] = '\0';
	}
	putchar('\n');
	fputs(&spaces[nbSpaces - level * INDENT_TAB], stdout);
    }
    free(comb);
}

/* concat a dirname and a basename in dst
 * dst is allocated by caller
 * try to be clever:
 * - add a  if needed t
 * dev + null -> devnull
 * - don't concat if basename start with /
 * dev + null -> null
 */
void
UConcatPath(char *dst, char *dirname, char *basename)
{
    int ldn;

    if (!(dirname && basename && dst))
	UIError("UConcatPath", "one parameter is NULL!");
    if (basename[0] == '/')
	strcpy(dst, basename);
    else {
	ldn = strlen(dirname);
	strcpy(dst, dirname);
	if (dirname[ldn - 1] == '/')
	    strcpy(dst + ldn, basename);
	else {
	    dst[ldn] = '/';
	    strcpy(dst + ldn + 1, basename);
	}
    }
}



/* ODOT ...
 */
int
UOpenPath(List * lPath, char *fileName, int flag)
{
    char tmp[MAX_CHAR_TMP];
    int i, openRes;

    char *pathComp;

    for (i = 0; i < lNbElts(lPath); i++) {
	pathComp = (char *) lLookNth(lPath, i);
	UConcatPath(tmp, pathComp, fileName);
	if (!access(tmp, F_OK)) {
	    openRes = open(tmp, flag);
	    return openRes;
	}
    }
    UIWarning("UOpenPath", "can't find file %s", fileName);
    return -2;
}
