#include <math.h>

#include "util.h"



/**************************************************************** Prototypes */


/******************************************************************** Bodies */



/* ODOT...
 */
Point3 *
V3FillPoint(Point3 * p, double x, double y, double z)
{
    p->x = x;
    p->y = y;
    p->z = z;
    return (p);
}

/* ODOT...
 */
Vector3 *
V3FillVector(Vector3 * v, double x, double y, double z)
{
    v->x = x;
    v->y = y;
    v->z = z;
    return (v);
}


/* ODOT...
 */
Point3 *
V3CopyPoint(Point3 * pin, Point3 * pout)
{
    pout->x = pin->x;
    pout->y = pin->y;
    pout->z = pin->z;
    return (pout);
}


/* ODOT...
 */
Vector3 *
V3CopyVector(Vector3 * vin, Vector3 * vout)
{
    vout->x = vin->x;
    vout->y = vin->y;
    vout->z = vin->z;
    return (vout);
}


/* print vector pv1
 */
void
V3PrintVector(Vector3 * pv1)
{
    printf("vect : %g %g %g", pv1->x, pv1->y, pv1->z);
}

/* compare vectors pv1 and pv2
 * useful for debugging
 * return 1 if they are identical else return 0 and print difference
 */
int
V3SameVector(Vector3 * pv1, Vector3 * pv2)
{
    if ((pv1->x != pv2->x) || (pv1->y != pv2->y) || (pv1->z != pv2->z)) {
	printf("vect diff: %g %g %g\n", pv2->x - pv1->x, pv2->y - pv1->y, pv2->z - pv1->z);
	return 0;
    }
    else
	return 1;
}

/* returns squared length of input vector */
double
V3SquaredLength(Vector3 * a)
{
    return ((a->x * a->x) + (a->y * a->y) + (a->z * a->z));
}


/* returns length of input vector */
double
V3Length(Vector3 * a)
{
    return (sqrt(V3SquaredLength(a)));
}


/* normalizes the input vector and returns it */
Vector3 *
V3Normalize(Vector3 * v)
{
    double len = V3Length(v);

    if (len != 0.0) {
	v->x /= len;
	v->y /= len;
	v->z /= len;
    }
    return (v);
}


/* return vector sum c = a+b */
Vector3 *
V3Add(Vector3 * a, Vector3 * b, Vector3 * c)
{
    c->x = a->x + b->x;
    c->y = a->y + b->y;
    c->z = a->z + b->z;
    return (c);
}


/* return vector difference c = a-b */
Vector3 *
V3Sub(Vector3 * a, Vector3 * b, Vector3 * c)
{
    c->x = a->x - b->x;
    c->y = a->y - b->y;
    c->z = a->z - b->z;
    return (c);
}


/* multiply each component of v by d */
Vector3 *
V3Scale(Vector3 * v, double d)
{
    v->x *= d;
    v->y *= d;
    v->z *= d;
    return (v);
}


/* return the dot product of vectors a and b */
double
V3Dot(Vector3 * a, Vector3 * b)
{
    return ((a->x * b->x) + (a->y * b->y) + (a->z * b->z));
}


/* return the cross product c = a cross b */
Vector3 *
V3Cross(Vector3 * a, Vector3 * b, Vector3 * c)
{
    c->x = (a->y * b->z) - (a->z * b->y);
    c->y = (a->z * b->x) - (a->x * b->z);
    c->z = (a->x * b->y) - (a->y * b->x);
    return (c);
}


/* return a vector constructed with two points */
Vector3 *
V3Points2Vector(Point3 * p1, Point3 * p2, Vector3 * v)
{
    v->x = p2->x - p1->x;
    v->y = p2->y - p1->y;
    v->z = p2->z - p1->z;
    return (v);
}


/* return the distance between two points */
double
V3DistanceBetween2Points(Point3 * a, Point3 * b)
{
    double dx = a->x - b->x;
    double dy = a->y - b->y;
    double dz = a->z - b->z;

    return (sqrt((dx * dx) + (dy * dy) + (dz * dz)));
}


/* multiply a point by a matrix and return the transformed point */
Point3 *
V3MulPointByMatrix(Point3 * pin, Matrix4 * m, Point3 * pout)
{
    pout->x = (pin->x * m->elt[0][0]) + (pin->y * m->elt[0][1]) +
    (pin->z * m->elt[0][2]);
    pout->y = (pin->x * m->elt[1][0]) + (pin->y * m->elt[1][1]) +
	(pin->z * m->elt[1][2]);
    pout->z = (pin->x * m->elt[2][0]) + (pin->y * m->elt[2][1]) +
	(pin->z * m->elt[2][2]);
    return (pout);
}


/* multiply a point by a projective matrix and return the transformed point */
Point3 *
V3MulPointByProjMatrix(Point3 * pin, Matrix4 * m, Point3 * pout)
{
    double w;

    pout->x = (pin->x * m->elt[0][0]) + (pin->y * m->elt[0][1]) +
	(pin->z * m->elt[0][2]) + m->elt[0][3];
    pout->y = (pin->x * m->elt[1][0]) + (pin->y * m->elt[1][1]) +
	(pin->z * m->elt[1][2]) + m->elt[1][3];
    pout->z = (pin->x * m->elt[2][0]) + (pin->y * m->elt[2][1]) +
	(pin->z * m->elt[2][2]) + m->elt[2][3];
    w = (pin->x * m->elt[3][0]) + (pin->y * m->elt[3][1]) +
	(pin->z * m->elt[3][2]) + m->elt[3][3];
    if ((w != 0.0) && (w != 1.0)) {
	pout->x /= w;
	pout->y /= w;
	pout->z /= w;
    }
    return (pout);
}

/* fill matrix mout with vectors c0 c1 c2 (used as columns)
 */
Matrix4 *
V3VectorToMat(Vector3 * c0, Vector3 * c1, Vector3 * c2, Matrix4 * mout)
{
    mout->elt[0][0] = c0->x;
    mout->elt[1][0] = c0->y;
    mout->elt[2][0] = c0->z;
    mout->elt[0][1] = c1->x;
    mout->elt[1][1] = c1->y;
    mout->elt[2][1] = c1->z;
    mout->elt[0][2] = c2->x;
    mout->elt[1][2] = c2->y;
    mout->elt[2][2] = c2->z;
    return mout;
}


/* multiply together matrices c = ab
 * note that c must not point to either of the input matrices
 */
Matrix4 *
V3MatMul(Matrix4 * a, Matrix4 * b, Matrix4 * c)
{
    int i, j, k;

    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    c->elt[i][j] = 0;
	    for (k = 0; k < 4; k++)
		c->elt[i][j] += a->elt[i][k] * b->elt[k][j];
	}
    }
    return (c);
}


/* multiply together matrices c = a * bt
 * note that c must not point to either of the input matrices
 * useful for invers of orthogonal matrix
 */
Matrix4 *
V3MatTransp2Mul(Matrix4 * a, Matrix4 * b, Matrix4 * c)
{
    int i, j, k;

    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    c->elt[i][j] = 0;
	    for (k = 0; k < 4; k++)
		c->elt[i][j] += a->elt[i][k] * b->elt[j][k];
	}
    }
    return (c);
}

/* multiply together matrices c = at * b
 * note that c must not point to either of the input matrices
 * useful for invers of orthogonal matrix
 */
Matrix4 *
V3MatTranspMul(Matrix4 * a, Matrix4 * b, Matrix4 * c)
{
    int i, j, k;

    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    c->elt[i][j] = 0;
	    for (k = 0; k < 4; k++)
		c->elt[i][j] += a->elt[k][i] * b->elt[k][j];
	}
    }
    return (c);
}

/* multiply all element of matrix by d */
Matrix4 *
V3MatNorm(Matrix4 * m, double d)
{
    int i, j;

    for (i = 0; i < 4; i++)
	for (j = 0; j < 4; j++)
	    m->elt[i][j] *= d;
    return (m);
}


/* determinant of m */
double
V3MatDet(Matrix4 * m)
{
    return (m->elt[0][0] * m->elt[1][1] * m->elt[2][2] * m->elt[3][3] - m->elt[0
									       ][0] * m->elt[1][1] * m->elt[2][3] * m->elt[3][2] - m->elt[0][0] * m->elt[2][1]
	    * m->elt[1][2] * m->elt[3][3] + m->elt[0][0] * m->elt[2][1] * m->elt[1][3] * m->
	    elt[3][2] + m->elt[0][0] * m->elt[3][1] * m->elt[1][2] * m->elt[2][3] - m->elt[0
											   ][0] * m->elt[3][1] * m->elt[1][3] * m->elt[2][2] - m->elt[1][0] * m->elt[0][1]
	    * m->elt[2][2] * m->elt[3][3] + m->elt[1][0] * m->elt[0][1] * m->elt[2][3] * m->
	    elt[3][2] + m->elt[1][0] * m->elt[2][1] * m->elt[0][2] * m->elt[3][3] - m->elt[1
											   ][0] * m->elt[2][1] * m->elt[0][3] * m->elt[3][2] - m->elt[1][0] * m->elt[3][1]
	    * m->elt[0][2] * m->elt[2][3] + m->elt[1][0] * m->elt[3][1] * m->elt[0][3] * m->
	    elt[2][2] + m->elt[2][0] * m->elt[0][1] * m->elt[1][2] * m->elt[3][3] - m->elt[2
											   ][0] * m->elt[0][1] * m->elt[1][3] * m->elt[3][2] - m->elt[2][0] * m->elt[1][1]
	    * m->elt[0][2] * m->elt[3][3] + m->elt[2][0] * m->elt[1][1] * m->elt[0][3] * m->
	    elt[3][2] + m->elt[2][0] * m->elt[3][1] * m->elt[0][2] * m->elt[1][3] - m->elt[2
											   ][0] * m->elt[3][1] * m->elt[0][3] * m->elt[1][2] - m->elt[3][0] * m->elt[0][1]
	    * m->elt[1][2] * m->elt[2][3] + m->elt[3][0] * m->elt[0][1] * m->elt[1][3] * m->
	    elt[2][2] + m->elt[3][0] * m->elt[1][1] * m->elt[0][2] * m->elt[2][3] - m->elt[3
											   ][0] * m->elt[1][1] * m->elt[0][3] * m->elt[2][2] - m->elt[3][0] * m->elt[2][1]
	    * m->elt[0][2] * m->elt[1][3] + m->elt[3][0] * m->elt[2][1] * m->elt[0][3] * m->
	    elt[1][2]);
}

/* return a nul matrix */
Matrix4 *
V3MatNul(Matrix4 * m)
{
    int i, j;

    for (i = 0; i < 4; i++)
	for (j = 0; j < 4; j++)
	    m->elt[i][j] = 0.0;
    return (m);
}


/* copy min in mout */
Matrix4 *
V3MatCopy(Matrix4 * min, Matrix4 * mout)
{
    int i, j;

    for (i = 0; i < 4; i++)
	for (j = 0; j < 4; j++)
	    mout->elt[i][j] = min->elt[i][j];
    return (mout);
}


/* place numbers on the diagonal line of the matrix */
Matrix4 *
V3MatDiag(Matrix4 * m, double d0, double d1, double d2, double d3)
{
    m->elt[0][0] = d0;
    m->elt[1][1] = d1;
    m->elt[2][2] = d2;
    m->elt[3][3] = d3;
    return (m);
}


/* return id matrix */
Matrix4 *
V3MatId(Matrix4 * m)
{
    V3MatNul(m);
    V3MatDiag(m, 1.0, 1.0, 1.0, 1.0);
    return (m);
}


/* return a translate matrix */
Matrix4 *
V3MatTranslate(Matrix4 * m, double deltaX, double deltaY, double deltaZ)
{
    V3MatId(m);
    m->elt[0][3] = deltaX;
    m->elt[1][3] = deltaY;
    m->elt[2][3] = deltaZ;
    return (m);
}


/*return a scale matrix */
Matrix4 *
V3MatScale(Matrix4 * m, double scaleX, double scaleY, double scaleZ)
{
    V3MatNul(m);
    V3MatDiag(m, scaleX, scaleY, scaleZ, 1.0);
    return (m);
}


/* return a rotate matrix for the x coordinates */
Matrix4 *
V3MatRotX(Matrix4 * m, double angle)
{
    V3MatNul(m);
    V3MatDiag(m, 1.0, cos(angle), cos(angle), 1.0);
    m->elt[1][2] = -sin(angle);
    m->elt[2][1] = sin(angle);
    return (m);
}


/* return a rotate matrix for the y coordinates */
Matrix4 *
V3MatRotY(Matrix4 * m, double angle)
{
    V3MatNul(m);
    V3MatDiag(m, cos(angle), 1.0, cos(angle), 1.0);
    m->elt[0][2] = sin(angle);
    m->elt[2][0] = -sin(angle);
    return (m);
}


/* return a rotate matrix for the z coordinates */
Matrix4 *
V3MatRotZ(Matrix4 * m, double angle)
{
    V3MatNul(m);
    V3MatDiag(m, cos(angle), cos(angle), 1.0, 1.0);
    m->elt[0][1] = -sin(angle);
    m->elt[1][0] = sin(angle);
    return (m);
}


/* return a shear rotate matrix */
Matrix4 *
V3MatShearXY(Matrix4 * m, double shx, double shy)
{
    V3MatId(m);
    m->elt[0][2] = shx;
    m->elt[1][2] = shy;
    return (m);
}


/* return a perspective-projection matrix */
Matrix4 *
V3MatPer(Matrix4 * m, double d)
{
    V3MatNul(m);
    V3MatDiag(m, 1.0, 1.0, 0.0, 1.0);
    m->elt[3][2] = -1.0 / d;
    return (m);
}


/* return a clip matrix */
Matrix4 *
V3MatClip(Matrix4 * m, double zm)
{
    V3MatNul(m);
    V3MatDiag(m, 1.0, 1.0, 1.0 / (1 + zm), 0.0);
    m->elt[2][3] = -zm / (1 + zm);
    m->elt[3][2] = -1;
    return (m);
}

/* compute the rotation matrix rm given v1 and v2
 * v2=rm*v1
 */
Matrix4 *
V3VectToRot(Vector3 * pv1, Vector3 * pv2, Matrix4 * rm)
{
    Vector3 n, v1, v2, dummy, vtmp, vtmp2;
    double cosTheta, sinTheta, omCosTheta;

    v1 = *pv1;
    v2 = *pv2;
    V3Normalize(&v1);
    V3Normalize(&v2);
    V3Cross(&v1, &v2, &n);
    V3Normalize(&n);

    cosTheta = V3Dot(&v1, &v2);
    sinTheta = V3Length(V3Cross(&v1, &v2, &dummy));
    omCosTheta = 1.0 - cosTheta;

    rm->elt[0][0] = cosTheta + (n.x * n.x) * omCosTheta;
    rm->elt[0][1] = n.x * n.y * omCosTheta - n.z * sinTheta;
    rm->elt[0][2] = n.x * n.z * omCosTheta + n.y * sinTheta;
    rm->elt[0][3] = 0.0;

    rm->elt[1][0] = n.y * n.x * omCosTheta + n.z * sinTheta;
    rm->elt[1][1] = cosTheta + (n.y * n.y) * omCosTheta;
    rm->elt[1][2] = n.y * n.z * omCosTheta - n.x * sinTheta;
    rm->elt[1][3] = 0.0;

    rm->elt[2][0] = n.z * n.x * omCosTheta - n.y * sinTheta;
    rm->elt[2][1] = n.z * n.y * omCosTheta + n.x * sinTheta;
    rm->elt[2][2] = cosTheta + (n.z * n.z) * omCosTheta;
    rm->elt[2][3] = 0.0;

    rm->elt[3][0] = 0.0;
    rm->elt[3][1] = 0.0;
    rm->elt[3][2] = 0.0;
    rm->elt[3][3] = 1.0;

    return rm;
}
