#include <stdio.h>
#include <X11/Xlib.h>
#include <values.h>
#include <malloc.h>
#include <stdlib.h>

#include "util.h"

#define NB_BEZIER_STEPS 50

char *XEventNames[] =
{
    "XError",
    "XReply",
    "KeyPress",
    "KeyRelease",
    "ButtonPress",
    "ButtonRelease",
    "MotionNotify",
    "EnterNotify",
    "LeaveNotify",
    "FocusIn",
    "FocusOut",
    "KeymapNotify",
    "Expose",
    "GraphicsExpose",
    "NoExpose",
    "VisibilityNotify",
    "CreateNotify",
    "DestroyNotify",
    "UnmapNotify",
    "MapNotify",
    "MapRequest",
    "ReparentNotify",
    "ConfigureNotify",
    "ConfigureRequest",
    "GravityNotify",
    "ResizeRequest",
    "CirculateNotify",
    "CirculateRequest",
    "PropertyNotify",
    "SelectionClear",
    "SelectionRequest",
    "SelectionNotify",
    "ColormapNotify",
    "ClientMessage",
    "MappingNotify",
    "LASTEvent",
};

/**************************************************************** Prototypes */


/******************************************************************** Bodies */


/* Convert a XEvent.type to its human readable form
 *
 * Return: the string corresponding to the type or "Unknown XEvent Type"
 */
char *
UXEventTypeToString(int type)
{
    if ((type < 0) || (type > sizeof(XEventNames) / sizeof(char *)))
	return "Unknown XEvent Type";

    else
	return XEventNames[type];
}

/* Compute rgb distance between two colors.
 */
int
UXRGBDist(XColor * c1, XColor * c2)
{
    return abs(c1->red - c2->red) + abs(c1->green - c2->green) + abs(c1->blue - c2->blue);
}

/* Create a GC with fore and bacground inverted.
 */
GC
UXInvertGc(Display * dpy, Drawable d, GC srcGc)
{
    GC invGc;
    XGCValues gcv;
    unsigned long tmpground;

    XGetGCValues(dpy, srcGc, (GCForeground | GCBackground), &gcv);
    tmpground = gcv.foreground;
    gcv.foreground = gcv.background;
    gcv.background = tmpground;
    invGc = XCreateGC(dpy, d, (GCForeground | GCBackground), &gcv);
    return invGc;
}



/* Find the nearest color in the default colormap of screen.
 */
int
UXFindNearestColor(Display * dpy, XColor * colRef, int *dist)
{
    Colormap defcm;
    XColor *qcols;
    Screen *screen;
    int nbCells;
    int bestDist, tmpd;
    int bestCol;
    int i;

    screen = ScreenOfDisplay(dpy, DefaultScreen(dpy));
    defcm = DefaultColormapOfScreen(screen);
    nbCells = CellsOfScreen(screen);
    qcols = (XColor *) malloc(nbCells * sizeof(XColor));
    for (i = 0; i < nbCells; i++) {
	qcols[i].pixel = i;
    }
    XQueryColors(dpy, defcm, qcols, nbCells);
    bestDist = MAXINT;
    for (i = 0; i < nbCells; i++) {
	tmpd = UXRGBDist(colRef, &qcols[i]);
	if (tmpd < bestDist) {
	    bestDist = tmpd;
	    bestCol = i;
	}
    }
    *dist = bestDist;
    return bestCol;
}

/* ODOT...
 */
void
UXReduce4Ximage(Display * dpy, XImage * srci, XImage * desti, int ulx, int uly, int w, int h, int *greyMap)
{
    register int val, sx, sy, dx, dy;
    int black;

    black = greyMap[4];
    for (dx = 0; dx < (w >> 1); dx++) {
	sx = dx << 1;
	for (dy = 0; dy < (h >> 1); dy++) {
	    sy = dy << 1;
	    val = 0;
	    if (XGetPixel(srci, sx, sy) == black)
		val++;
	    sx++;
	    if (XGetPixel(srci, sx, sy) == black)
		val++;
	    sy++;
	    if (XGetPixel(srci, sx, sy) == black)
		val++;
	    sx--;
	    if (XGetPixel(srci, sx, sy) == black)
		val++;

	    XPutPixel(desti, dx, dy, greyMap[val]);
	}
    }
}

/* Draw a bezier curve given 4 control points.
 */
void
UXDrawBezier(Display * dpy, Drawable d, GC gc, int x0, int y0, int x1, int y1, int x2, int y2, int x3, int y3)
{
    float dt, ax, bx, cx, ay, by, cy, t = 0.0;
    int nsteps = NB_BEZIER_STEPS;
    int px1, py1, px2, py2;
    char first = 1;

    dt = 1.0 / nsteps;
    cx = 3.0 * (x1 - x0);
    bx = 3.0 * (x2 - x1) - cx;
    ax = x3 - (x0 + cx + bx);
    cy = 3.0 * (y1 - y0);
    by = 3.0 * (y2 - y1) - cy;
    ay = y3 - (y0 + cy + by);

    while (t <= 1.0 + dt / 10.0) {
	px1 = px2;
	py1 = py2;
	px2 = ((ax * t + bx) * t + cx) * t + x0;
	py2 = ((ay * t + by) * t + cy) * t + y0;
	if (!first) {
	    XDrawLine(dpy, d, gc, px1, py1, px2, py2);
	}
	else {
	    first = 0;
	}
	t += dt;
    }
}
