#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/time.h>
#include <string.h>
#include <signal.h>

#include "comm.h"
#include "util.h"

/* must rebuil external fd_set? useful for selects...
 * each connect deconect increment it by one
 */
int USMustRebuildFDSet = 1;

static GArray *aServConn = NULL;	/* array of connections with apps */
static int sockfd;			/* socket file desciptor */


/**************************************************************** Prototypes */

/* Close the socket descriptor
 */
static void
     USCloseSocketFd(void);

/******************************************************************** Bodies */



/* Close the socket descriptor
 */
static void
USCloseSocketFd(void)
{
    close(sockfd);
}

/* Init the server.
 *
 * Must be called before any call to other function of this lib.
 *
 * Return: 0 or error code
 *
 * Warning: this function set the signal handler for SIG_PIPE to SIG_IGN
 */
int
USInit(int portNum, char *serverName)
{
    struct sockaddr_in address;


    UTraceSetLevel(TRACE_UCOMM);
    if ((sockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	return errno;

    address.sin_family = AF_INET;	/* Using Internet domain */
    address.sin_port = htons(portNum);	/* ???? why htons? */
    address.sin_addr.s_addr = 0;
    if (bind(sockfd, (struct sockaddr *) & address, sizeof(address)) < 0) {
	close(sockfd);
	return errno;
    }

    if (listen(sockfd, 8) < 0) {	/* 8 is current limit of backlog */
	close(sockfd);
	return errno;
    }

#ifdef __osf__
    atexit(USCloseSocketFd);		/* force to close socket on exit */
#endif

#ifdef __ultrix__
    atexit(USCloseSocketFd);		/* force to close socket on exit */
#endif

#ifdef __sun__
    on_exit(USCloseSocketFd, NULL);	/* force to close socket on exit */
#endif

    signal(SIGPIPE, SIG_IGN);		/* ignore broken sockets */

    return 0;
}

/* Close the line (id).
 *
 * Return: 0 or error code
 */
int
USCloseLine(int id)
{
    USMustRebuildFDSet++;
    return UCloseLineCommon(id, aServConn);
}

/* Check if new applications are requesting for lines.
 *
 * Accept the first one and put it's id in pId.
 * Use pGetLine (if non NULL) to return if got Line (0 or 1)
 * (must be called until *pGetLine is 0).
 *
 * Return: 0 or error code
 *
 * Warning: this function must be called periodicaly to insure that connection
 * requests aren't ignored
 */
int
USAcceptNewLine(int *pId, int *pGotLine)
{
    fd_set readfds, writefds, exceptfds;
    struct timeval timeout;
    struct sockaddr_in caller;
    int socketNum, dummy;
    ConnectionLine *pConn;

    if (pGotLine)
	*pGotLine = 0;
    (timeout).tv_sec = 0;
    (timeout).tv_usec = 0;
    /* Looking for new clients */
    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    FD_ZERO(&exceptfds);
    FD_SET(sockfd, &readfds);
    if (URestartSelect(sockfd + 1, (fd_set *) & readfds, (fd_set *) & writefds,
		       (fd_set *) & exceptfds, &timeout) < 0) {
	close(sockfd);
	return errno;
    }
    dummy = sizeof(struct sockaddr_in);

    if (FD_ISSET(sockfd, &readfds)) {	/* New client asking for connection */
	if ((socketNum = accept(sockfd, (struct sockaddr *) & caller, &dummy)) < 0) {	/* accepting connection */
	    return errno;
	}
	pConn = UNew(ConnectionLine);
	pConn->lineName = NULL;
	pConn->socketNum = socketNum;
	pConn->lReceivedMsg = lCreate((void *(*) ()) UKeyRcvMsg, strcmp, UFreeRcvMsg, UPrintRcvMsg);
	bzero(pConn->packToSend, UCOMM_PACKET_SIZE);
	pConn->packToSendLen = 0;
	pConn->nbSendPacket = 0;
	pConn->nbSendMsg = 0;
	pConn->nbReceivedMsg = 0;
	*pId = UAddAConn(pConn, &aServConn);
	USMustRebuildFDSet++;
	if (pGotLine)
	    *pGotLine = 1;
	URetIfNz(UIncWaitingPacketsCommon(pConn));
    }
    return 0;
}


/* Wait for a new application.
 *
 * Accept it and put it's id in pId.
 *
 * Warning: this function may block.
 */
int
USWaitForNewLine(int *pId)
{
    fd_set readfds, writefds, exceptfds;
    struct sockaddr_in caller;
    int socketNum, dummy;
    ConnectionLine *pConn;

    /* Looking for new clients */
    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    FD_ZERO(&exceptfds);
    FD_SET(sockfd, &readfds);
    if (URestartSelect(sockfd + 1, (fd_set *) & readfds, (fd_set *) & writefds,
		     (fd_set *) & exceptfds, (struct timeval *) NULL) < 0) {
	close(sockfd);
	return errno;
    }
    dummy = sizeof(struct sockaddr_in);

    if (FD_ISSET(sockfd, &readfds)) {	/* New client asking for connection */
	if ((socketNum = accept(sockfd, (struct sockaddr *) & caller, &dummy)) < 0) {	/* accepting connection */
	    return errno;
	}
	pConn = UNew(ConnectionLine);
	pConn->lineName = NULL;
	pConn->socketNum = socketNum;
	pConn->lReceivedMsg = lCreate((void *(*) ()) UKeyRcvMsg, strcmp, UFreeRcvMsg, UPrintRcvMsg);
	bzero(pConn->packToSend, UCOMM_PACKET_SIZE);
	pConn->packToSendLen = 0;
	pConn->nbSendPacket = 0;
	pConn->nbSendMsg = 0;
	pConn->nbReceivedMsg = 0;
	*pId = UAddAConn(pConn, &aServConn);
	USMustRebuildFDSet++;
	URetIfNz(UIncWaitingPacketsCommon(pConn));
    }
    return 0;
}

/* Send a message (on at least store it in a buffer).
 *
 * Return: 0 or error code
 */
int
USSend(int id /* On which line to send msg */ ,
       void *msg /* Pointer on data to send */ ,
       int len /* Lenght of data to send */ )
{
    return USendCommon((ConnectionLine *) gaLookNth(aServConn, id), UCOMM_MSG_USER, msg, len);
}

/* Send all messages which may have been buffered (if any)
 * (ie flush actual packet (if needed)).
 *
 * Return: 0 or error code
 */
int
USFlush(int id)
{
    return UFlushCommon((ConnectionLine *) gaLookNth(aServConn, id));
}

/* Flush messages waiting to be send for all clients
 */
int
USFlushAllClients()
{
    int nbe, i, stat;

    nbe = gaSize(aServConn);
    for (i = 0; i < nbe; i++) {
	if ((stat = USFlush(i)))
	    return stat;
    }
    return 0;
}

/* How many messages are waiting for a read.
 *
 * Return imediatly if messages are already queued.
 * If no message queued, flush buffer and wait for roundtrip
 * (ie be sure no message are kept in other line end buffer).
 *
 * Warning: function may block a bit if correspondant is too busy.
 */
int
USPendingMsg(int id, int *pNbMsg)
{
    return UPendingMsgCommon((ConnectionLine *) gaLookNth(aServConn, id), pNbMsg);
}


/* How many messages are waiting for a read.
 *
 * Checks for waiting packets and return
 * nether blocks
 */
int
USCheckPendingMsg(int id, int *pNbMsg)
{
    return UCheckPendingMsgCommon((ConnectionLine *) gaLookNth(aServConn, id), pNbMsg);
}

/* Checks if there is something waiting for a read on all connected lines
 *
 * Do the check as fast as possible (only one select)
 * nether blocks
 */
int
USCheckAllLines(int *pIsMsg)
{
    return UCheckAllLinesCommon(aServConn, pIsMsg);
}

/* Fill a fd_set (eg: for a select) with sockets fd of all limes connected
 * to the server.
 *
 * Return: max fd stored in *pFdSet
 */
int
USFillFdAllLines(fd_set * pFdSet)
{
    int maxFd;

    /* fill with all lines fd */
    maxFd = UFillFdAllLinesCommon(aServConn, pFdSet);
    /* and add listening socket to detect a connect */
    FD_SET(sockfd, pFdSet);
    return Max(maxFd, sockfd);
}


/* Get next message.
 *
 * If no message in waiting queue;
 * Flush this line-end.
 * Make periodics roundtrip to insure no messages are in buffer at the
 * other line-end.
 *
 * Return: 0 or error code.
 *
 * Warning: It's a blocking call.
 */
int
USGetNextMsg(int id,
	     char **ppMsg		/* Pointer on returned message
	         (caller must free ppMsg after use) */ ,
	     int *pLen /* Lenght of returned message */ )
{
    return UGetNextMsgCommon((ConnectionLine *) gaLookNth(aServConn, id), ppMsg, pLen);
}

/* Peek next message (message is not removed from queue).
 *
 * If no message in waiting queue;
 * Flush this line-end.
 * Fake periodics roundtrip to insure no messages are in buffer at the
 * other line-end.
 *
 * Return: 0 or error code
 *
 * Warning: It's a blocking call.
 */
int
USPeekNextMsg(int id,
	      char **ppMsg		/* Pointer on returned message
					 * (caller must NOT free ppMsg after
	          use) */ ,
	      int *pLen /* Lenght of returned message */ )
{
    return UPeekNextMsgCommon((ConnectionLine *) gaLookNth(aServConn, id), ppMsg, pLen);
}

/* Return (and remove) first matching message from queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 */
int
USGetFirstMatchingMsg(int id,
		      char *head	/* String which must be the head of
		          matching message */ ,
		      char **ppMsg,
		      int *pLen)
{
    return UGetFirstMatchingMsgCommon((ConnectionLine *) gaLookNth(aServConn, id), head, ppMsg, pLen);
}

/* Return (but don't remove) first matching message form queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 */
int
USPeekFirstMatchingMsg(int id,
		       char *head	/* String which must be the head of
		           matching message */ ,
		       char **ppMsg,
		       int *pLen)
{
    return UPeekFirstMatchingMsgCommon((ConnectionLine *) gaLookNth(aServConn, id), head, ppMsg, pLen);
}

/* Return id associated to line nammed (lineName)
 * or -1 if no such line
 */
int
USNameToId(char *lineName)
{
    return UNameToIdCommon(lineName, aServConn);
}

/* Return name of line identified by id
 * or NULL if no such line
 *
 * Warning: Caller is responsible for freeing the returned string
 */
char *
USIdToName(int id)
{
    return UIdToNameCommon(id, aServConn);
}

/* Checks if this name may be given to this line.
 *
 * Actually just checks if it has not been set for another line.
 *
 * Return: False on problem or True if it's ok.
 *
 * Warning: runs only on server side.
 */
Bools
USCheckForGoodName(char *name)
{
    int i, nbc;
    GArray *aConn;
    ConnectionLine *pConn;

    aConn = aServConn;
    nbc = gaSize(aConn);
    for (i = 0; i < nbc; i++) {
	pConn = gaLookNth(aConn, i);
	if ((pConn) && (pConn->lineName)) {
	    if (!strcmp(pConn->lineName, name))
		return False;
	}
    }
    return True;
}

/* Convert an error code to a string.
 *
 * CWarning: Caller is responsible for freeing the returned string.
 */
char *
USErrCodeToErrString(int errCode)
{
    return UErrCodeToErrString(errCode);
}
