#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>
#include <sys/errno.h>			/* for error code during open */

#include "util.h"

/* trace level used by nmacros UTrace* */
long UTraceLevel;


/**************************************************************** Prototypes */


/******************************************************************** Bodies */


/* A wrapper for exit.
 * If NO_EXIT is defined any call to UExit is fatal...
 * segmentation fault (core dumped)!!
 * Useful when debuginig.
 */
void
UExit(int status)
{
#ifdef NO_EXIT
    fflush(stdout);
    MAKECORE				/* What a nice core dump!!! */
#endif
	exit(status);
}


/* Example:  UIWarning should be called like:
 * UIWarning(function_name, format, arg1, arg2...);
 * the format string is like this used by printf
 * UIWarning("CheckedMalloc", "can't allocate %d bytes, %d allocated"
 *           ,size, allocatedSize);
 */
void
UIWarning(char *funcName, char *format,...)
{
    va_list args;

    va_start(args, format);
    /* print name of function causing warning */
    (void) fprintf(stderr, "Internal WARNING in %s: ", funcName);
    /* print out remainder of message */
    (void) vfprintf(stderr, format, args);
    (void) putc('\n', stderr);
    va_end(args);
}


/* Example: UIError should be called like:
 * UIError(function_name, format, arg1, arg2...);
 * the format string is like this used by printf
 * UIError("CheckedMalloc", "can't allocate %d bytes, %d allocated"
 *         ,size, allocatedSize);
 */
void
UIError(char *funcName, char *format,...)
{
    va_list args;

    va_start(args, format);
    /* print name of function causing warning */
    (void) fprintf(stderr, "Internal ERROR in %s: ", funcName);
    /* print out remainder of message */
    (void) vfprintf(stderr, format, args);
    (void) putc('\n', stderr);
    va_end(args);
    UExit(1);
}



/* Given a open-file-error-number build the error message string
 */
char *
UOpenError(int err, char *f)
{
    char tmp[MAX_CHAR_TMP];
    char tmp2[MAX_CHAR_TMP];

    switch (err) {
    case EACCES:{
	    sprintf(tmp2, "file not found or permision denied (EACCES)");
	    break;
	}
    case EDQUOT:{
	    sprintf(tmp2, "quota overflow (EDQUOT)");
	    break;
	}
    case EEXIST:{
	    sprintf(tmp2, "file already exist (EEXIST)");
	    break;
	}
    case EFAULT:{
	    sprintf(tmp2, "mem error (EFAULT)");
	    break;
	}
    case EINTR:{
	    sprintf(tmp2, "signal caught during open (EINTR)");
	    break;
	}
    case EIO:{
	    sprintf(tmp2, "hangup error (EIO)");
	    break;
	}
    case EISDIR:{
	    sprintf(tmp2, "it's a directory (EISDIR)");
	    break;
	}
    case ELOOP:{
	    sprintf(tmp2, "too many symbolic links (ELOOP)");
	    break;
	}
    case EMFILE:{
	    sprintf(tmp2, "too much files have already been opened by this process (EMFILE)");
	    break;
	}
    case ENAMETOOLONG:{
	    sprintf(tmp2, "too long path name (ENAMETOOLONG)");
	    break;
	}
    case ENFILE:{
	    sprintf(tmp2, "too much files have already been opened (ENFILE)");
	    break;
	}
    case ENOENT:{
	    sprintf(tmp2, "file does not exist (ENOENT)");
	    break;
	}
    case ENOSPC:{
	    sprintf(tmp2, "disk full (ENOSPC)");
	    break;
	}
    case ENOTDIR:{
	    sprintf(tmp2, "component of the path prefix is not a directory (ENOTDIR)");
	    break;
	}
    case ENXIO:{
	    sprintf(tmp2, "(ENXIO)");
	    break;
	}
    case EROFS:{
	    sprintf(tmp2, "read only file system (EROFS)");
	    break;
	}
    default:{
	    sprintf(tmp2, "unknown error (%d)", err);
	    break;
	}
    }
    sprintf(tmp, "can't open file %s: %s", f, tmp2);
    return UStrDup(tmp);
}
