/* Copyright 1989-93 GROUPE BULL -- See license conditions in file COPYRIGHT */
/**************\
*              *
*  KlO  Stream *
*  BODY        *
*              *
\**************/

#include "EXTERN.h"
#include <signal.h>
#include <fcntl.h>
#ifdef SYSV
#include <unistd.h>
#endif /* SYSV */
#include <sys/types.h>
#include <sys/time.h>     
#include <sys/stat.h>
#ifdef NEED_SELECT_H
#include <sys/select.h>
#endif
#include "klone.h"
#include "kl_number.h"
#include "kl_atom.h"
#include "kl_list.h"
#include "kl_string.h"
#include "kl_func.h"
#include "klgeneric.h"
#include "INTERN.h"
#include "kl_stream.h"

#include <errno.h>

KlActive KlA_stdin;
KlActive KlA_stdout;
KlActive KlA_stderr;
KlConstant KlA_StdinOrig;
KlConstant KlA_StdoutOrig;
KlConstant KlA_StderrOrig;
KlO KlStreamToStringCoerce();

/* direction of files:
 * 0 = closed
 * 1 = read
 * 2 = write
 * ==> 3 read&write
 */

char *KlStreamDirectionsText[] = {"closed", "read", "write", "read&write"};

/* constructor for Files
 */

KlStream
KlStreamMake(fd, direction, name)
    FILE *fd;
    int direction;
    char *name;
{
    KlStream obj;

    obj = (KlStream) KlOMake(KlStreamType);
    obj->subtype = KlStreamFileType;
    obj->fd = fd;
    obj->write_fd = 0;
    obj->direction = direction;
    obj->lineno = 1;
    obj->unput[0] = '\0';
    obj->name = (char *) Malloc(strlen(name) + 1);
    strcpy(obj->name, name);
    obj->is_reading_file = obj->name;
    obj->blocking = 1;
    obj->blocking_write = 1;

    return obj;
}

/* constructor for Strings
 */

KlStreamString
KlStreamStringMake(klstring, direction)
    KlString klstring;
    int direction;
{
    KlStreamString obj;

    obj = (KlStreamString) KlOMake(KlStreamType);
    obj->subtype = KlStreamStringType;
    KlIncRef(obj->klstring = klstring);
    obj->direction = direction;
    obj->lineno = 1;
    obj->unput[0] = '\0';
    obj->cursor = 0;
#ifndef USE_STANDARD_MALLOC
    obj->limit = (klstring->string ? KlMallocedSize(klstring->string) : 0);
#else
    obj->limit = (klstring->string ? KlModStringLength(klstring) + 1 : 0);
#endif

    return obj;
}

/* KlOpen
 * opening a new stream
 * (open filename &key :direction :if-exists :type :error) [646]
 * 
 *     creates a stream by opening the file filename. keywords can be:
 *     :direction    :output, :input (default), or :io (both)
 *     :if-exists    :overwrite, :supersede or :append (default) 
 *		  what to do if file exists.
 * 		  already and we open it in :output or :io mode)
 *     :type	  :file (default) or :string. The string type is a string that
 * 		  can be written to and read from by I/O primitives, but can
 * 		  be still used as a string.
 *     :buffered  can be t (default) or nil
 *     :writer    file to open for writing, if :direction :io and write channel
 *		  is different from read channel
 *     :blocking  can be t (default) or (), in which case FD_NDELAY is set
 *     :error     if an error happens, returns the evaluation of argument
 *                instead of calling KlE_ERROR_OPENING_FILE. Note that since
 *                open evaluates its arguments, you need to quote the 
 *                expression
 */

/* list of valid parameters to keywords. see at the end of this file */
static KlKeyword
    *KlOpenKV_if_exists,
    *KlOpenKV_direction,
    *KlOpenKV_filetype;

KlStream
KlOpenError(error, name, mode)
    KlO error;
    KlString name;
    KlKeyword mode;
{
    if (error) {
	return (KlStream) KlSend_eval(error);
    } else {
	return (KlStream) KlError2(KlE_ERROR_OPENING_FILE, name, mode);
    }
}

KlStream
KlOpen(argc, argv)
    int argc;
    KlO *argv;
{
    KlO direction, if_exists, filetype, buffered, blocking, error;
    KlString writer;
    KlString filename;
    int no_delay;

    KlParseKeywords(argc, argv, 1);
    direction = KlKeyVal(KlK_direction, KlK_input);
    if_exists = KlKeyVal(KlK_if_exists, KlK_append);
    filetype = KlKeyVal(KlK_type, KlK_file);
    buffered = KlKeyVal(KlK_buffered, 0);
    writer = (KlString) KlKeyVal(KlK_writer, 0);
    blocking =  KlKeyVal(KlK_blocking, TRU);
    error = KlKeyVal(KlK_error, 0);
    KlCheckUnvalidKeywords(argc, argv, 1);

    /* check the validity of the arguments to the keywords. */
    if_exists = KlCheckKeywordValue(KlK_if_exists, if_exists,
				    KlOpenKV_if_exists);
    direction = KlCheckKeywordValue(KlK_direction, direction,
				    KlOpenKV_direction);
    filetype = KlCheckKeywordValue(KlK_type, filetype,
				   KlOpenKV_filetype);
    no_delay = KlFalseP(blocking);

    KlMustBeString(argv[0], 0);
    filename = (KlString) argv[0];

    if (filetype == (KlO) KlK_string) {	/* string stream */
	KlStreamString stream;

	KlMustBeAModifiableString(argv[0], 0);
	stream =
	    KlStreamStringMake(filename,
			       direction == (KlO) KlK_output ? 2 :
			       direction == (KlO) KlK_io ? 3 :
			       1);
	
	if (direction == (KlO) KlK_output
	    || direction == (KlO) KlK_io) {
	    if (if_exists == (KlO) KlK_append) {
		stream->cursor = KlModStringLength(stream->klstring);
	    } else if (if_exists == (KlO) KlK_supersede) {
		stream->klstring->string[0] = '\0';
                KlModStringSetLength(stream->klstring, 0);
	    }
	    /* overwrite ==> cursor = 0 */
	}
	return (KlStream) stream;
    } else {				/* default: file stream */
	KlStream stream;
	char *mode;
	FILE *fd, *write_fd = 0;

	KlMustBeString(argv[0], 0);
	if (direction == (KlO) KlK_output) {
	    mode = (if_exists == (KlO) KlK_overwrite ? "r+" :
		if_exists == (KlO) KlK_supersede ? "w" : "a");
	} else if (direction == (KlO) KlK_io) {
	    mode = (if_exists == (KlO) KlK_overwrite ? "r+" :
		if_exists == (KlO) KlK_supersede ? "w+" : "a+");
	    if (writer) {
		KlMustBeString(writer, KlPosition(argc, argv, KlK_writer) + 1);
		if (writer->string[0] == '~') {
		    writer = KlExpandTildeForFiles(writer->string);
		}
		if (!(write_fd = fopen(writer->string, mode))) {
		    return KlOpenError(error, writer, KlK_output);
		}
		mode = "r";
	    }
	} else {			/* input */
	    mode = "r";
	}

	if (!(fd = fopen(
		    (filename->string[0] == '~'
			? KlExpandTildeForFiles(filename->string)->string
			: filename->string),
		    mode))) {
	    return  KlOpenError(error, filename,
				write_fd ? KlK_input : (KlKeyword) direction);
	}
	stream = KlStreamMake(fd,
	    direction == (KlO) KlK_output ? 2 :
	    direction == (KlO) KlK_io ? 3 :
	    1,
	    filename->string);
	if (write_fd) {
	    stream->write_fd = write_fd;
	}
	if (buffered == NIL) {		/* unbuffer stream */
	    setbuf(((stream->direction & KlStreamOWRITE) && write_fd
		    ? write_fd
		    : fd),
		   0);
	}
	if (direction == (KlO) KlK_output
	    && if_exists == (KlO) KlK_overwrite) {
	    fseek(fd, 0L, 0);
	    if (write_fd) {
		fseek(write_fd, 0L, 0);
	    }
	}
	if (no_delay) {
	    KlSetFILENoBlocking(fd, 1, &(stream->blocking));
	    if (writer)
		KlSetFILENoBlocking(write_fd, 1, &(stream->blocking_write));
	}
	return stream;
    }
}

/* auxilliary function */

#ifdef SYSV
#define KlNDELAY O_NDELAY
#else
#define KlNDELAY FNDELAY
#endif

KlSetFILENoBlocking(fp, flag, klone_flag)
    FILE *fp;
    int flag;
    char *klone_flag;
{
    int file_mode;

    fcntl(KlFp2Fd(fp), F_GETFL, &file_mode);
    if (flag)
	file_mode |= KlNDELAY;	/* set non-blocking flag */
    else
	file_mode &= ~KlNDELAY;	/* remove non-blocking flag */
    fcntl(KlFp2Fd(fp), F_SETFL, file_mode);
    *klone_flag = flag;
}

/*****************************************************************************\
* 				 KlStreamMode                                 *
\*****************************************************************************/
/* stream-mode stream
 * :writer t|()
 * :blocking t|()
 * :buffered t|()			only buffered () does something
 * (stream-mode [:writer flag]) lists modes
 * (stream-mode :buffered ()) MUST be done before first read or write
 */
    
KlO
KlStreamMode(argc, argv)
    int argc;
    KlO *argv;
{
    KlO writer, blocking, buffered;
    KlStream stream;
    FILE *fp;
    char *flagp;


    KlParseKeywords(argc, argv, 1);
    writer =  KlKeyVal(KlK_writer, NIL);
    blocking =  KlKeyVal(KlK_blocking, 0);
    buffered = KlKeyVal(KlK_buffered, 0);
    KlCheckUnvalidKeywords(argc, argv, 1);

    stream = (KlStream) argv[0];
    KlMustBeStream(stream, 0);

    if (stream->subtype & KlStreamFileType) {
	if (KlTrueP(writer) && stream->write_fd) {
	    fp = stream->write_fd;
	    flagp = &(stream->blocking_write);
	} else {
	    fp = stream->fd;
	    flagp = &(stream->blocking);
	}

	if (blocking) {			/* set */
	    KlSetFILENoBlocking(fp, KlFalseP(blocking), flagp);
	}
	if (buffered && KlFalseP(buffered)) {
	    setbuf(fp, 0);
	}
	if (!blocking && !buffered) {			/* get */
	    if (KlTrueP(writer) && stream->write_fd) {
		return (KlO) KlListPairMake(KlK_blocking,
					    stream->blocking_write
					    ? TRU : NIL);
	    } else {
		return (KlO) KlListPairMake(KlK_blocking,
					    stream->blocking ? TRU : NIL);
	    }
	}
    }
    return (KlO) stream;
}

/*****************************************************************************\
* 				   KlFseek                                    *
\*****************************************************************************/
/* KlFseek
 * seeking into a file 
 * (file-position stream &optional offset from_where)
 * from_where = 0,1,2 for start,current_pos,end of file
 * other values = ftell (default)
 */

KlO
KlFseek(argc, argv)
    int argc;
    KlO *argv;
{
    KlStreamString stream;
    int offset = 0;
    int from_where = -1;

    switch(argc) {
    case 1:
	break;
    case 2:
	KlMustBeNumber(argv[1], 1);
	offset = ((KlNumber) argv[1])->number;
	from_where = 0;
	break;
    case 3:
	KlMustBeNumber(argv[1], 1);
	KlMustBeNumber(argv[2], 2);
	offset = ((KlNumber) argv[1])->number;
	from_where = ((KlNumber) argv[2])->number;
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }

    stream = (KlStreamString)  argv[0];
    KlMustBeStream(stream, 0);

    if (stream->subtype & KlStreamFileType) {
	if (((unsigned int) from_where) > 2) {
	    return (KlO) KlNumberMake(ftell(((KlStream) stream)->fd));
	} else {
	    fseek(((KlStream) stream)->fd, offset, from_where);
	}
    } else {
	int len = KlModStringLength(stream->klstring);
	int pos;

	switch (from_where) {
	case 0:
	    pos = offset;
	    break;
	case 1:
	    pos = stream->cursor + offset;
	    break;
	case 2:
	    pos = len + offset;
	    break;
        default:
	    return (KlO) KlNumberMake(stream->cursor);
	}
	if (pos >= 0) {
	    if (pos >= stream->limit) {
		stream->limit = KlMallocChunkSize(pos + 1);
		stream->klstring->string = (char *)
		    Realloc(stream->klstring->string, stream->limit);
	    }
	    stream->cursor = pos;
	    if (pos > len) {		/* if past the end, fill with blanks */
		bzero(stream->klstring->string+len, pos-len);
		stream->klstring->string[pos] = '\0';
		KlModStringSetLength(stream->klstring, pos);
	    }
	}
    }
    return NIL;
}

/* printing streams
 */

KlO
KlStreamPrint(obj, stream)
    KlStream obj;
    KlStream stream;
{
    KlSPrintf(stream, "{^ %s", KlTypeCName(obj->type));
    KlSPrintf(stream, " 0x%x ", obj);
    if (obj->subtype & KlStreamFileType) {
	KlSPrintf(stream, "\"%s\"", obj->name);
	KlSPrintf(stream, " #%d", KlFp2Fd(obj->fd));
    } else {
	KlSend_print(KlA_String, stream);
    }
    KlSPrintf(stream, " %s}", KlStreamDirectionsText[obj->direction]);
    return (KlO) obj;
}

/* closing streams
 * (close stream [ :writer t|() ] )
 * writer allows you to close only one direction of a bidirectional stream
 */

KlO
KlStreamClose(argc, argv)
    int argc;
    KlO *argv;
{
    KlStream obj;
    KlO writer;

    KlParseKeywords(argc, argv, 1);
    writer =  KlKeyVal(KlK_writer, 0);
    KlCheckUnvalidKeywords(argc, argv, 1);
    obj = (KlStream) argv[0];
    KlMustBeStream(obj, 0);

    /* nothing is needed to be done for strings */
    /* we do not send kill to the process */
    if (obj->subtype & KlStreamFileType) {
	if (obj->direction) {
	    signal(SIGPIPE, SIG_IGN);
	    if (writer && obj->direction == 3) {
		if (KlTrueP(writer)) {	/* close output dir */
		    if (obj->write_fd) {
			fclose(obj->write_fd);
			obj->write_fd = 0;
		    }
		    obj->direction = 1;
		} else {		/* close input dir */
                    if (obj->write_fd) {
			fclose(obj->fd);
			obj->fd = obj->write_fd;
			obj->write_fd = 0;
		    }
		    obj->direction = 2;
		}
		KlTrapSIGPIPE();
		return NIL;
	    } else {
		fclose(obj->fd);
		obj->fd = 0;
		if (obj->write_fd) {
		    fclose(obj->write_fd);
		    obj->write_fd = 0;
		}
	    }
	    KlTrapSIGPIPE();
	}
    }
    obj->direction = 0;
    return NIL;
}

/* freeing streams (closes files)
 */

KlO
KlStreamFree(obj)
    KlStream obj;
{
    if (obj->subtype == KlStreamStringType) { /* strings */
	KlDecRef(((KlStreamString) obj)->klstring);
    } else {				/* file & processes */
	KlStreamClose(1, &obj);
	Free(obj->name);
    }
    Free(obj);
    return (KlO) obj;
}

/*ARGSUSED*/
KlResetStdout(old, dummy)
    KlStream old;
    KlO dummy;
{
    KlStdout = old;
}

/*****************************************************************************\
* 				IO redirection                                *
\*****************************************************************************/
/* active values on:
 * *standard-input* [497]
 * *standard-output*
 * *standard-error*
 */

KlO
KlStreamStdGet(stptr)
    KlStream *stptr;
{
    return (KlO) * stptr;
}

KlO
KlStreamStdSet(obj, stptr)
    KlStream obj;
    KlStream *stptr;
{
    KlMustBeStream(obj, 0);
    if (stptr == &KlStdin) {
	KlStreamCanREAD(obj);
    } else {
	KlStreamCanWRITE(obj);
	KlFlush(0);
    }

    KlDecRef(*stptr);
    KlIncRef(*stptr = obj);

    return (KlO) obj;
}

/*****************************************************************************\
* 				parsing stream                                *
\*****************************************************************************/

extern int KlyyinIsString;
extern char *Klyystrin;
extern char *Klyysptr, Klyysbuf[];
extern FILE *Klyyin;

/* to redirect parsing stream
 * returns old stream (or 0 if not changed)
 */

KlStream
KlStdyyRedirect(stream)
    KlStream stream;
{
    KlStream oldstream;

    if (!KlStdyy) {
	KlIncRef(KlStdyy = KlStdin);
    }
    if ((stream != KlStdyy) && stream) {
	KlStreamCanREAD(stream);
	if (oldstream = KlStdyy) {
	    KlDecRef(KlStdyy);
	    KlyySave();
	}
	KlIncRef(KlStdyy = stream);
	KlyyRestore();
	return oldstream;
    } if ((stream == KlStdyy) && stream	
	&& (stream->subtype == KlStreamStringType)) {
	/* refresh stream string: used when changing cursor to make
	 * parser re-start reading at some place (after a seek)
	 */
	KlStdyy->unput[0] = '\0';	/* flush read-ahead */
	KlyyRestore();			/* adjust to cursor value */

	return stream;
    } else {
	return 0;
    }
}


/* save/restore of lex/yacc context for stream
 */

KlyySave()
{
    if (KlStdyy->subtype == KlStreamStringType) {
	((KlStreamString) KlStdyy)->cursor = Klyystrin -
	    ((KlStreamString) KlStdyy)->klstring->string;
    } else {
	KlStdyy->is_reading_file = KlIsReadingFile;
    }

    if (Klyysptr != Klyysbuf)
	strncpy(KlStdyy->unput, Klyysbuf, Klyysptr - Klyysbuf);
    KlStdyy->unput[Klyysptr - Klyysbuf] = '\0';
    Klyysptr = Klyysbuf;
    KlStdyy->lineno = Klyylineno;
}

KlyyRestore()
{
    if (KlStdyy->subtype == KlStreamStringType) {
	KlyyinIsString = 1;
	Klyystrin = ((KlStreamString) KlStdyy)->klstring->string
	    + ((KlStreamString) KlStdyy)->cursor;
	KlIsReadingFile = 0;
    } else {
	KlyyinIsString = 0;
	Klyyin = KlStdyy->fd;
	KlIsReadingFile = KlStdyy->is_reading_file;
    }
    {
	char *p = KlStdyy->unput;

	Klyysptr = Klyysbuf;
	while (*p)
	    *Klyysptr++ = *p++;
    }
    Klyylineno = KlStdyy->lineno;
}

/*
 * KlRead: is now a macro
 * reads an expression from the input (string or stream).
 * returns this expression or NULL if EOF reached
 * In case of syntax error, returns NIL
 * the read expression is in the global variable KlReadExpr,
 * if you need it. (this global is maintained for ref count purposes)
 * You don't need to free it since it's done at the beginning of this
 * routine.
 * Beware that it could be overwritten by a subsequent call to KlEval
 * or KlRead !
 */

/* The klone-callable read function
 * jumps to tag EOF on EOF
 */

KlO
KlReadKl(argc, argv)
    int argc;
    KlO *argv;
{
    KlStream oldstream = 0;

    if (argc) {
	KlMustBeStream(argv[0], 0);
	KlStreamCanREAD(((KlStream)argv[0]));
	if (argc > 2)
	    return KlBadNumberOfArguments(argc);
	oldstream = KlStdyyRedirect(argv[0]);
    }
    KlUnwindProtectStatement(Klyyparse(),  KlApplyUnary, KlStdyyRedirect,
			     oldstream);
    if (KlReadExpr)
	return KlReadExpr;
    else
	if (argc >= 2)
	    return KlSend_eval(argv[1]);
	else
	    KlThrow(KlA_EOF, NIL);		/* EOF */
    /* NOTREACHED */
}

/* KlParseString
 * parses a C string and returns expr. (with execute parameter 0)
 * raw no-error-checking function to be used in C initialisations
 */

KlO
KlParseStringRaw(execute)
    int execute;			/* 0= only parse, 1= real/+eval loop */
{
    KlO result = NIL;
    if (execute) {
	while (KlRead()) {
	    result = KlSend_eval(KlReadExpr);
	}
    } else {
	Klyyparse(), result = KlReadExpr ? KlReadExpr : NIL;
    }
    return result;
}

KlO
KlParseString(l, s, execute)
    int l;				/* length of string */
    char *s;
    int execute;                        /* 0= only parse, 1= real/+eval loop */
{
    KlStream oldstream = 0;
    KlO result;

    oldstream = KlStdyyRedirect(KlStreamStringMake(KlStringMakeNoCopy(l, s),
						   1));
    KlUnwindProtectStatement(result = KlParseStringRaw(execute), KlApplyUnary,
			     KlStdyyRedirect, oldstream);
    return result;
}

/*****************************************************************************\
* 				      IO                                      *
\*****************************************************************************/

/* flushing a stream
 * 0 = stdout + stderr
 */

KlO
KlFlush(obj)
    KlStream obj;
{
    if (obj && KlTrueP(obj)) {
	if (obj->type == KlStreamType
	    && (obj->subtype & KlStreamFileType)
	    && (obj->direction & KlStreamOWRITE)) {
	    fflush(KlStreamWriteFd(obj));
	}
    } else {
	if (KlStdout->subtype & KlStreamFileType
	    && KlStdout->direction & KlStreamOWRITE)
	    fflush(KlStreamWriteFd(KlStdout));
	if (KlStderr->subtype & KlStreamFileType
	    && KlStderr->direction & KlStreamOWRITE)
	    fflush(KlStreamWriteFd(KlStderr));
    }
    return NIL;
}

/***********************\
* C- callable functions *
\***********************/

/* prints a printf-formatted string */

KlSPrintf(stream, format, string)
    KlStream stream;
    char *format;
    char *string;
{
    KlStreamCanWRITE(stream);
    if (stream->subtype == KlStreamStringType) {
	char tmp[KlMAX_TEMP_STRING_SIZE];

	sprintf(tmp, format, string);
	KlSPuts(tmp, stream);
    } else {
	fprintf(KlStreamWriteFd(stream), format, string);
    }
}

/* print-format
 * (print-format stream format arguments...)
 * %N are replaced by the printing of the N-th argument
 * for 0 <= N < 9
 * (type of object can be printed also by %tN)
 * %N can be printed readably by %rN
 * % is printed by %%
 * if format is not a string it is applied to arguments and the result is
 * used as format
 */
    
KlPrintFormatAux(stream, argc, argv)
    KlStream stream;
    int argc;
    KlO *argv;
{
    char *format, *p;
    int pos;
    int print_with_type;
    int print_readably;			/* 0=as is, 1= no, 2 = readably */
    KlString klformat = (KlString) argv[1];
    int prv = KlPrintReadably;

    if (klformat->type != KlStringType) { /* if not a string, apply */
	if (KlIsAnAtom(klformat) && KlIsAString(((KlAtom)klformat)->c_val)) {
	    klformat = (KlString) ((KlAtom)klformat)->c_val;
	} else {
	    KlList call = KlListKl(argc - 1, argv + 1);

	    klformat = (KlString) KlApply(call);
	    KlMustBeString(klformat, 1);
	}
    }
    format = (char *) KlAlloca(((KlStringLength(klformat) + 1) / KLSO) + 1);
    strcpy(format, klformat->string);
    for (p = format; *p; p++) {
	if (*p == '%') {
	    int option_char = 1;
	    *p++ = '\0';
	    KlSPuts(format, stream);
	    format = p+1;
	    print_with_type = 0;
	    print_readably = 0;
	    do {
		switch (*p) {
		case 'r' :
		    print_readably = 2;
		    break;
		case 'n':
		    print_readably = 1;
		    break;
		case 't':
		    print_with_type = 1;
		    break;
		default:
		    option_char = 0;
		}
		if (option_char) {
		    format++;
		    p++;
		}
	    } while (option_char);

	    if (*p >= '0' && *p <= '9') {
		pos = (*p) - '0' + 2;
		if (pos >= 2 && pos < argc) {
		    if (print_readably)
			KlPrintReadably = print_readably - 1;
		    if (print_with_type) {
			KlSPuts(KlTypeCName((argv[pos])->type), stream);
		    } else {
			KlSPrint(argv[pos], stream);
		    }
		    if (print_readably)
			KlPrintReadably = prv;
		}
	    } else {			/* %<any_char> = % */
		KlSPutc('%', stream);
	    }
	}
    }
    if (*format) {
	KlSPuts(format, stream);
    }
}

KlO
KlPrintFormat(argc, argv)
    int argc;
    KlO *argv;
{
    KlStream stream;

    KlNumberOfArgumentsCheck(argc < 1, argc);
    /* decode the polymorph first argument */
    if (KlIsAString(argv[0])
	       && !KlIsASymbol(argv[0])) { /* stream arg can be omitted */
	stream = KlStdout;	
	argc++;
	argv--;
    } else if (KlIsAStream(argv[0])) {	/* normal case, stream */
	stream = (KlStream) argv[0];
	KlStreamCanWRITE(stream);
    } else if ((argv[0] == (KlO) KlStringType) /* create new string stream */
	       || (argv[0] == (KlO) KlA_String)) {
	KlStreamString klstream = KlStreamStringMake(KlStringMake(0), 3);
	KlPrintFormatAux(klstream, argc, argv);
	return (KlO) klstream->klstring;
    } else if (argv[0] == NIL) {		/* () = stdout */
	stream = KlStdout;
    } else {
	KlMustBeStream(argv[0], 0);	/* give meaningful error mess */
    }
    KlPrintFormatAux(stream, argc, argv);
    return NIL;
}

/* KlSPutBytes
 * the heart of printing in Klone. Nearly all output goes through this.
 */

KlSPutBytes(length, buffer, stream)
    int length;
    char *buffer;
    KlStreamString stream;
{
    if (stream->subtype == KlStreamStringType) {
	if ((stream->cursor + length) >= stream->limit) {	/* realloc */
	    stream->limit = KlMallocChunkSize(stream->cursor + length + 1);
	    stream->klstring->string = (char *)
		Realloc(stream->klstring->string, stream->limit);
	    stream->klstring->string[stream->cursor + length] = '\0';
	}
	bcopy(buffer, stream->klstring->string + stream->cursor, length);
	stream->cursor += length;
	if (stream->cursor >= KlModStringLength(stream->klstring)) {
	    /* we extended the string, let's add a null byte for C compat */
	    KlModStringSetLength(stream->klstring, stream->cursor);
	    stream->klstring->string[stream->cursor] = '\0';
	}
    } else {
	if (length !=
	    fwrite(buffer, 1, length, KlStreamWriteFd((KlStream) stream))) {
	    KlError2(KlE_STREAM_ERROR, KlA_write, stream);  /* no space left */
	}
    }
}

/* KlSPuts
 * prints a string. Mainly used from short C strings now that 
 * Klone strings are byte stings
 */

KlSPuts(string, stream)
    char *string;
    KlStreamString stream;
{
    if (stream->subtype == KlStreamStringType) {
	KlSPutBytes(strlen(string), string, stream);
    } else {
	if (EOF == fputs(string, (KlStreamWriteFd((KlStream) stream)))) {
	    /* error: no space left */
	    KlError2(KlE_STREAM_ERROR, KlA_write, stream);
	}
    }
}

/* reading a string
 * returns pointer to static storage or 0 on EOF
 * reads till \n or eof, do not returns final \n.
 * mistakes NULL bytes in the input for EOF!, but this provides mucho speed!
 */

static int KlGets_tmpl;
static char *KlGets_tmp;

char *
KlGets(stream, lenptr)
    KlStream stream;
    int *lenptr;			/* returns length of string */
{
    int length;

    if (stream->subtype == KlStreamStringType) { /* string */
	KlStreamString klstream = (KlStreamString) stream;
	char *start = klstream->klstring->string + klstream->cursor;
	char *p = start;

	if (!*p)
	    return 0;
	while (*p && *p != '\n')
	    p++;
	length = p - start;
	if (length >= KlGets_tmpl) {
	    KlGets_tmpl = length;
	    KlGets_tmp = (char *) Realloc(KlGets_tmp, KlGets_tmpl + 1);
	}
	bcopy(start, KlGets_tmp, length);
	KlGets_tmp[length] = '\0';
	klstream->cursor = p - klstream->klstring->string + ((*p) ? 1 : 0);
	*lenptr = length;
	return KlGets_tmp;
    } else {				/* file */
	return KlFGets(stream->fd, lenptr);
    }
}

/* KlFGets
 * useful function to read an unlimited line (but terminates on '\0')
 */

char *
KlFGets(fd, lenptr)
    FILE *fd;
    int *lenptr;
{
    char *p = KlGets_tmp;
    int length;

    KlGets_tmp[KlGets_tmpl - 2] = '\0'; /* marker overriden when more */
    while (fgets(p, KlGets_tmpl - (p - KlGets_tmp), fd)) {
	if (KlGets_tmp[KlGets_tmpl - 2]
	    && KlGets_tmp[KlGets_tmpl - 2] != '\n') { /* more to read */
	    KlGets_tmpl += 1024;
#ifdef VOIDPTR_MALLOC
	    KlGets_tmp = (char *) Realloc(KlGets_tmp, KlGets_tmpl);
#else
	    KlGets_tmp = Realloc(KlGets_tmp, KlGets_tmpl);
#endif /* VOIDPTR_MALLOC */
	    p = KlGets_tmp + KlGets_tmpl - 1 - 1024;
	    KlGets_tmp[KlGets_tmpl - 2] = '\0';
	} else {			/* done */
	    goto done;
	}
    }
    if (p == KlGets_tmp)		/* EOF */
	return 0;
 done:
    /* trim last newline if present and return */
    length = strlen(KlGets_tmp);
    if (length > 0 && KlGets_tmp[length - 1] == '\n')
	KlGets_tmp[--length] = '\0';
    else
	KlGets_tmp[length] = '\0';
    *lenptr = length;
    return KlGets_tmp;
}

/* reading a char
 * retunrs EOF on eof
 */

int
KlGetc(stream)
    KlStream stream;
{
    if (stream->subtype == KlStreamStringType) {
	KlStreamString klstream = (KlStreamString) stream;
	char *p = klstream->klstring->string + klstream->cursor;

	if (*p) {
	    klstream->cursor++;
	    return *p;
	} else {
	    return EOF;
	}
    } else {
	return getc(stream->fd);
    }
}

/*************************\
* klone-callable functions *
\*************************/

KlResetKlPrintLevel(a, b)
    KlO a, b;
{
    KlPrintLevel = -1;
}

/*
 *  ? (or print) simply print value of an object
 */
/* klone-level one,  handling *print-level*
 */

KlO
KlPrintNary(argc, argv)
    int argc;
    KlO argv[];
{
    int i;

    if (KlPrintLevel >= 0) {		/* we are inside a print */
	for (i = 0; i < argc; i++)
	    KlSend_print(argv[i], KlStdout);
    } else {				/* we start printing, init p-l */
	KlPrintLevel = 0;
	KlUnwindProtectStatement
	    (for (i = 0; i < argc; i++) KlSend_print(argv[i], KlStdout),
	     KlResetKlPrintLevel, 0, 0);
    }
    KlFlush(0);
    return argc ? argv[argc - 1] : NIL;
}

/* low-level one, do not handle *print-level*
 */

KlO
KlWrite(argc, argv)
    int argc;
    KlO *argv;
{
    switch (argc) {
    case 1:
	KlSend_print(argv[0], KlStdout);
	break;
    case 2:
	KlMustBeStream(argv[1], 1);
	KlStreamCanWRITE(((KlStream) argv[1]));
	KlSend_print(argv[0], argv[1]);
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }
    return argv[0];
}

/* klone-level one,  handling *print-level*
 */

KlO
KlWriteKl(argc, argv)
    int argc;
    KlO *argv;
{
    if (KlPrintLevel >= 0) {		/* we are inside a print */
	return KlWrite(argc, argv);
    } else {				/* we start printing, init p-l */
	KlO result;
	KlPrintLevel = 0;
	KlUnwindProtect(KlWrite(argc, argv), result, KlResetKlPrintLevel,
			0, 0);
	return result;
    }
}

KlO
KlWriteChar(argc, argv)
    int argc;
    KlO *argv;
{
    KlStream stream;

    switch (argc) {
    case 1:
	stream = KlStdout;
	break;
    case 2:
	stream = (KlStream) argv[1];
	KlMustBeStream(stream, 1);
	KlStreamCanWRITE(stream);
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }
    if (KlIsAString(argv[0])) {
	KlSPuts(((KlString) argv[0])->string, stream);
    } else {
	KlMustBeNumber(argv[0], 0);
	KlSPutc(((KlNumber) argv[0])->number, stream);
    }
    return argv[0];
}

KlO
KlWriteLine(argc, argv)
    int argc;
    KlString *argv;
{
    KlStream stream;

    switch (argc) {
    case 1:
	stream = KlStdout;
	break;
    case 2:
	stream = (KlStream) argv[1];
	KlMustBeStream(stream, 1);
	KlStreamCanWRITE(stream);
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }
    KlMustBeString(argv[0], 0);
    KlSPutBytes(KlStringLength(argv[0]), argv[0]->string, stream);
    KlSPutc('\n', stream);
    return (KlO) argv[0];
}

/* read functions callable from klone */

KlO
KlReadChar(argc, argv)
    int argc;
    KlStream *argv;
{
    KlStream stream;
    int c;

    switch (argc) {
    case 0:
	stream = KlStdin;
	break;
    case 1: case 2:
	stream = argv[0];
	KlMustBeStream(stream, 0);
	KlStreamCanREAD(stream);
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }
    c = KlGetc(stream);
    if (c == EOF) {
	if (argc == 2)
	    return KlSend_eval(argv[1]);
	else
	    KlThrow(KlA_EOF, NIL);
	/* NOTREACHED */
    } else {
	return (KlO) KlNumberMake(c);
    }
}

KlO
KlReadLine(argc, argv)
    int argc;
    KlStream *argv;
{
    KlStream stream;
    char *s;
    int len;

    switch (argc) {
    case 0:
	stream = KlStdin;
	break;
    case 1: case 2:
	stream = argv[0];
	KlMustBeStream(stream, 0);
	KlStreamCanREAD(stream);
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }
    if (s = KlGets(stream, &len)) {
	return (KlO) KlStringMakeFromBytes(len, s);
    } else {
	if (argc == 2)
	    return KlSend_eval(argv[1]);
	else
	    KlThrow(KlA_EOF, NIL);
	/* NOTREACHED */
    }
}

/*****************************************************************************\
*        readwrite of N chars at a time, on possibly non-blocking streams     *
\*****************************************************************************/

/* (read-chars N [stream])
*/

KlO
KlReadChars(argc, argv)
    int argc;
    KlStream *argv;
{
    KlStream stream = KlStdin;
    int n = -1;

    switch (argc) {
    case 2:
	stream = argv[1];
	KlMustBeStream(stream, 1);
	KlStreamCanREAD(stream);
	/* no break intentional */
    case 1:
	if (KlTrueP(argv[0])) {
	    KlMustBeNumber(argv[0], 0);
	    n = ((KlNumber) argv[0])->number;
	}
	/* no break intentional */
    case 0:
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }
    if (n >= 0) {
	if (stream->subtype == KlStreamStringType) { /* string */
	    int nread =
		Min(n, KlStringLength(((KlStreamString) stream)->klstring)
		    - ((KlStreamString) stream)->cursor);
	    return (KlO) KlStringMakeFromBytes
		(nread, ((KlStreamString) stream)->klstring
		 + ((KlStreamString) stream)->cursor);
	} else {			/* file */
	    KlString s;
	    char *buffer = (char *) Malloc(n + 1);
	    int nread = fread(buffer, 1, n, stream->fd);
	    buffer[nread] = '\0';
	    s = KlStringMakeFromBytes(nread, buffer);
	    Free(buffer);
	    return (KlO) s;
	}
    } else {				/* gobble up everything */
	return (KlO) KlStreamToStringCoerce(KlStringType, stream);
    }
}

/* (write-chars string N [stream] [offset])
 */

KlO
KlWriteChars(argc, argv)
    int argc;
    KlStream *argv;
{
    KlStream stream = KlStdout;
    KlString s;
    int n = -1;
    int offset = 0;

    switch (argc) {
    case 4:
	KlMustBeNumber(argv[3], 3);
	offset = ((KlNumber) argv[3])->number;
    case 3:
	stream = argv[2];
	KlMustBeStream(argv[2], 2);
	KlStreamCanWRITE(stream);
	/* no break intentional */
    case 2:
	if (KlTrueP(argv[1])) {
	    KlMustBeNumber(argv[1], 1);
	    n = ((KlNumber) argv[1])->number;
	}
	/* no break intentional */
    case 1:
	break;
    default:
	return KlBadNumberOfArguments(argc);
    }

    s = (KlString) argv[0];
    KlMustBeString(argv[0], 0);
    if (n >= 0)
	n = Min(n, KlStringLength(s) - offset);
    else
	n = KlStringLength(s) - offset;
    if (stream->subtype == KlStreamStringType) { /* no problemo */
	KlSPutBytes(n, s->string + offset, stream);
    } else {				/* file */
	n = fwrite(s->string + offset, 1, n,
		   KlStreamWriteFd((KlStream) stream));
    }
    return (KlO) KlNumberMake(n);
}

/**************************************************************************\
* 				    select                                 *
\**************************************************************************/
/* (select [:input] streams... [:output streams...] [:error streams]
 *         [:timeout milliseconds-or-nil])
 * works with buffering and strings!
 * returns () (no stream were ready, and timeout expired or signal received)
 * or a list of 3 lists being the lists of streams ready for input, output,
 * and having an exceptional condition pending.
 * streams can be streams or lists of streams which are expanded.
 * any stream can be () and is then ignored
 * :input keyword is implicitely delared at the start
 * :timeout 0 means non-blocking, nonexisting or () means blocks indefinitely
 *     (default)
 * keywords can appear in any order
 * select is made of :input (read), :output (write) or :error (execptional
 *     condition pending)
 * before going to the UNIX select() call observed streams are examined. If
 * one of them satisfies one of the following criteria, KlStreamSelect returns
 * immediately with the matching  streams without actually calling select(2)
 *  [1] buffered input file stream having some characters still in the input
 *      buffer in the FILE structure
 *  [2] string streams in write mode (always ready)
 *  [3] string streams in read mode, but not at EOF
 * WARNING: (select) blocks indefinitely!
 * WARNING: select on a file at EOF returns true!. to test if a file is a EOF,
 *          if select is true but read-non-blocking retunrs 0, then EOF is 
 *          here.
 */

/* descriptor used to prepare the arguments lists to select(2) */
struct KlStreamList {
    int mode;				/* read, write or read&write (error) */
    fd_set set;				/* to be passed to select */
    KlList list;			/* list of selected streams */
    KlList ready_list;			/* results: streams with pending IO */
};

KlO
KlStreamSelect(argc, argv)
    int argc;
    KlStream *argv;
{
    int nfds = 0, n, i, res;
    struct KlStreamList arglists[3], *arglistp = &(arglists[0]);
    struct timeval timeout, *timeoutp = 0; /* blocking */
    KlStream *arg = argv, *last = argv+argc;

    for (i = 0; i < 3; i++) {		/* init the 3 KlStreamLists */
	FD_ZERO(&(arglists[i].set));
	arglists[i].list = KlListNMake(0);
	arglists[i].ready_list = KlListNMake(0);
	arglists[i].mode = i+1;		/* hack */
    }

    while (arg < last) {		/* parses args */
	if (KlIsAKeyword(*arg)) {	/* keyword change cur. KlStreamList */
	    if (KlK_timeout == (KlKeyword) *arg) {
		if (++arg < last && KlTrueP(*arg)) {
		    KlMustBeNumber(*arg, arg - argv);
		    timeout.tv_sec = ((KlNumber)*arg)->number / 1000;
		    timeout.tv_usec = (((KlNumber)*arg)->number % 1000) * 1000;
#ifdef AMIGA
		    if (((KlNumber)*arg)->number == 0)
			timeout.tv_usec = 1; /* bug in amiga select? */
#endif /* AMIGA */
		    timeoutp = &timeout;
		} else {
		    timeoutp = 0;	/* () ==> no timeout, blocking */
		}
	    } else if (KlK_input == (KlKeyword) *arg) {
		arglistp = arglists;
	    } else if (KlK_output == (KlKeyword) *arg) {
		arglistp = arglists + 1;
	    } else if (KlK_error == (KlKeyword) *arg) {
		arglistp = arglists + 2;
	    } else {
		KlKeyword KlSelectKeywords[5], *p = KlSelectKeywords;
		*p++ = KlK_input;
		*p++ = KlK_output;
		*p++ = KlK_error;
		*p++ = KlK_timeout;
		*p++ = 0;
		KlError2(KlE_INVALID_KEYWORD, *arg,
			 KlListNullTerminated(KlSelectKeywords));
	    }
	} else if (KlIsAList(*arg)) {	/* expand lists of streams */
	    for (i = 0; i < ((KlList) (*arg))->size; i++) {
		if ((n = KlStreamSelectArgAdd
		     (((KlList)(*arg))->list[i], arg-argv, arglistp))
		     > nfds)
		    nfds = n;
	    }
	} else {			/* simple case of a streams */
	    if ((n = KlStreamSelectArgAdd(*arg, arg-argv, arglistp)) > nfds)
		nfds = n;
	}
	arg++;
    }
    for (i = 0; i < 3; i++)		/* returns if something in buffers */
	if (arglists[i].ready_list->size) 
	    goto end;

					/* do the UNIX call */
    for (;;) {
	KlLastSignal = 0;
	res = select(nfds, &(arglists[0].set), &(arglists[1].set), 
		     &(arglists[2].set), timeoutp);
	if (res > 0) {
	    /* OK, constructs the returned triplet */
	    KlStream obj;
	    for (i = 0; i < arglists[0].list->size; i++) {
		if (KlIsAFileStream(obj = (KlStream) arglists[0].list->list[i])
		    && FD_ISSET(KlFp2Fd(obj->fd), &(arglists[0].set)))
		    KlListAppend(arglists[0].ready_list, obj);
	    }
	    for (i = 0; i < arglists[1].list->size; i++) {
		if (KlIsAFileStream(obj = (KlStream) arglists[1].list->list[i])
		    && FD_ISSET(KlFp2Fd(KlStreamWriteFd(obj)),
				&(arglists[1].set)))
		    KlListAppend(arglists[1].ready_list, obj);
	    }
	    for (i = 0; i < arglists[2].list->size; i++) {
		if (KlIsAFileStream(obj = (KlStream) arglists[2].list->list[i])
		    && (FD_ISSET(KlFp2Fd(obj->fd), &(arglists[2].set))
			|| (obj->write_fd
			    && FD_ISSET(KlFp2Fd(obj->write_fd),
					&(arglists[2].set)))))
		    KlListAppend(arglists[2].ready_list, obj);
	    }
	    goto end;
	} else if (res < 0) {		/* error (signal such as SIGCHLD) */
	    if (errno != EINTR || KlLastSignal != KlSIGCHLD) 
		return NIL;		/* SIGCHLD ? continue : return NIL */
	} else {				/* none pending */
	    return NIL;
	}
    }
  end:
    {					/* actual building of returned list */
	KlList reslist = KlListNMake(3);
	for (i = 0; i < 3; i++)
	    KlIncRef(reslist->list[i] = (KlO) arglists[i].ready_list);
	return (KlO) reslist;
    }
}

/* main parsing (by accumulation) used by KlStreamSelect
 * accumulates arg into a KlStreamList struct
 * checks if a stream is structurally ready (chars in buffer) and already
 * puts it in ready_list field
 * return fd+1 if correct file descriptor monitored or 0
 */

int
KlStreamSelectArgAdd(obj, pos, slistp)
    KlStream obj;
    int pos;
    struct KlStreamList *slistp;
{
    int fd, res = 0;
    if (KlFalseP(obj))			/* ignore () */
	return 0;
    KlMustBeStream(obj, pos);
    if (!(obj->direction & (slistp->mode)))
	KlError2(KlE_STREAM_ERROR, (slistp->mode) ? KlA_write : KlA_read, obj);
    KlListAppend(slistp->list, obj);
    if (KlIsAFileStream(obj)) {		/* file */
	if ((obj->direction) & KlStreamOWRITE & (slistp->mode)) {
	    fd = KlFp2Fd(KlStreamWriteFd(obj));
	    FD_SET(fd, &(slistp->set));
	    res = Max(res, fd+1);
	}
	if ((obj->direction) & KlStreamOREAD & (slistp->mode)) { 
	    fd = KlFp2Fd(obj->fd);
	    FD_SET(fd, &(slistp->set));
	    res = Max(res, fd+1);
	}
	if (slistp->mode == KlStreamOREAD /* if chars in buffer, ready */
	    && READ_DATA_PENDING(obj->fd))
	    KlListAppend(slistp->ready_list, obj);
    } else {				/* string */
	if (slistp->mode == KlStreamOWRITE /* strings always ready to write */
	    || (slistp->mode == KlStreamOREAD
		&& ((KlStreamString)obj)->cursor <
		KlModStringLength(((KlStreamString)obj)->klstring->string)))
	    KlListAppend(slistp->ready_list, obj);
    }
    return res;
}


/*****************************************************************************\
* 				  coercions                                   *
\*****************************************************************************/

/* string->stream 
 * open a string stream
 */

/*ARGSUSED*/
KlO
KlStringToStreamCoerce(totype, obj)
    KlType totype;
    KlString obj;
{
    return (KlO) KlStreamStringMake(KlStringCopy(obj), 3);
}

/*ARGSUSED*/
KlO
KlStreamToStringCoerce(totype, stream)
    KlType totype;
    KlStream stream;
{
    struct stat buf;

    if (stream->subtype == KlStreamStringType) { /* string */
	return (KlO) ((KlStreamString)stream)->klstring;
    /* regular file of known length */
    } else if (!fstat(KlFp2Fd(stream->fd), &buf)
	       && (buf.st_mode & S_IFREG)) {
	int size;
	KlString str = KlStringNMake(buf.st_size);
	size = fread(str->string, 1, buf.st_size, stream->fd);
	*(str->string + size) = '\0';
	KlModStringSetLength(str, size);
	return (KlO) str;
    } else {				/* stream of unknown length */
	KlString str = KlStringNMake(1019);
	int offset = 0;
	int nitems;
	int to_get = 1019;
	int bufsize = 1024;
	
	while (nitems = fread(str->string + offset, 1, to_get,
			      stream->fd)) {
	    offset += nitems;
	    if (offset > bufsize / 2) {
		bufsize *= 2;
		str->string = (char *) Realloc(str->string, bufsize - 4);
	    }
	    to_get = bufsize - offset -4 -1;
	}
	*(str->string + offset) = '\0';
	KlModStringSetLength(str, offset);
	return (KlO) str;
    }
}

/*****************************************************************************\
* 				  TYPE INIT                                   *
\*****************************************************************************/

KlStreamInit()
{
    int i;

    KlDeclareType(&KlStreamType, "Stream", sizeof(struct _KlStream));

    KlDeclareMethod1(KlStreamType, KlSelPrint, KlStreamPrint);
    KlDeclareMethod1(KlStreamType, KlSelFree, KlStreamFree);

    KlDeclareSubr(KlOpen, "open", NARY);
    KlDeclareSubr(KlStreamClose, "close", NARY);
    KlA_write = (KlAtom) KlDeclareSubr(KlWrite, "write", NARY);
    KlDeclareSubr(KlWriteLine, "write-line", NARY);
    KlDeclareSubr(KlWriteChar, "write-char", NARY);
    KlA_read = (KlAtom) KlDeclareSubr(KlReadKl, "read", NARY);
    KlDeclareSubr(KlReadLine, "read-line", NARY);
    KlDeclareSubr(KlReadChar, "read-char", NARY);
    KlDeclareSubr(KlFlush, "flush", 1);
    KlDeclareSubr(KlPrintFormat, "print-format", NARY);
    KlDeclareSubr(KlFseek, "file-position", NARY);
    KlDeclareSubr(KlStreamMode, "stream-mode", NARY);
    KlDeclareSubr(KlReadChars, "read-chars", NARY);
    KlDeclareSubr(KlWriteChars, "write-chars", NARY);
    KlDeclareSubr(KlStreamSelect, "select", NARY);

    KlIncRef(KlStdin = KlStreamMake(stdin, 1, "stdin"));
    KlIncRef(KlStdout = KlStreamMake(stdout, 2, "stdout"));
    KlIncRef(KlStderr = KlStreamMake(stderr, 2, "stderr"));

    KlA_StdinOrig = KlConstantMake("*standard-input-orig*", KlStdin);
    KlA_StdoutOrig = KlConstantMake("*standard-output-orig*", KlStdout);
    KlA_StderrOrig = KlConstantMake("*standard-error-orig*", KlStderr);

    KlA_stdin = KlActiveMake("*standard-input*",
	KlStreamStdGet, KlStreamStdSet, &KlStdin);
    KlA_stdout = KlActiveMake("*standard-output*",
	KlStreamStdGet, KlStreamStdSet, &KlStdout);
    KlA_stderr = KlActiveMake("*standard-error*",
	KlStreamStdGet, KlStreamStdSet, &KlStderr);

    KlStdyy = 0;

    KlIncRef(KlStdpool = KlStreamStringMake(KlStringMake(" "), 1));
    
    KlDeclareCoerce(KlStringType, KlStreamType, KlStringToStreamCoerce);
    KlDeclareCoerce(KlStreamType, KlStringType, KlStreamToStringCoerce);

    /* keyword lists */
    
    KlOpenKV_if_exists = (KlKeyword *) Malloc(KLSO * 4);    i = 0;
    KlOpenKV_if_exists[i++] = KlK_append;
    KlOpenKV_if_exists[i++] = KlK_supersede;
    KlOpenKV_if_exists[i++] = KlK_overwrite;
    KlOpenKV_if_exists[i++] = 0;

    KlOpenKV_direction = (KlKeyword *) Malloc(KLSO * 4);    i = 0;
    KlOpenKV_direction[i++] = KlK_input;
    KlOpenKV_direction[i++] = KlK_io;
    KlOpenKV_direction[i++] = KlK_output;
    KlOpenKV_direction[i++] = 0;

    KlOpenKV_filetype = (KlKeyword *) Malloc(KLSO * 3);    i = 0;
    KlOpenKV_filetype[i++] = KlK_file;
    KlOpenKV_filetype[i++] = KlK_string;
    KlOpenKV_filetype[i++] = 0;
#ifdef AMIGA
    Klyyin = stdin;
#endif /* AMIGA */

    KlGets_tmpl = 1020;
    KlGets_tmp = (char *) Malloc(KlGets_tmpl);
}
