/* Copyright 1989-93 GROUPE BULL -- See license conditions in file COPYRIGHT */
/*********************\
* 		      *
*  KlO  Atom  *
*  BODY		      *
* 		      *
\*********************/

#include "EXTERN.h"
#include "klone.h"
#include "kl_coll.h"
#include "kl_number.h"
#include "kl_list.h"
#include "kl_string.h"
#include "kl_stream.h"
#include "INTERN.h"
#include "kl_atom.h"

#undef EXT				/* EXTERN */
#define EXT extern
#undef INIT
#define INIT(x)
#undef DO_INIT
#undef PRIVATE_DEFINITIONS

#include "kl_func.h"

#include "klversion.h"

extern KlO KlStringAdd();
extern KlO KlStringGet();
static int KlAtomBucket;

/*
 * Constructor:
 * Constructed via the hash table management routines.
 */

/*
 * Warning: You should NEVER call KlAtomMake, call KlIntern instead,
 * or your atom won't be in the hash table!
 */

KlAtom
KlAtomMake(p_name, c_val)		/* makes an atom */
    char *p_name;			/* is COPIED to atom's p_name */
    KlO c_val;				/* is just pointed to */
{
    /* we cannot use the handy KlOMake here, for bootstrapping problems */
    KlAtom object = (KlAtom) KlMallocBucket(KlAtomBucket);

    KlZrtPut(object);
    object->p_name = (char *) Malloc(strlen(p_name) + 1);
    strcpy(object->p_name, p_name);
    if (*p_name != ':') {		/* atom */
	object->type = KlAtomType;
	KlIncRef(object->c_val = c_val);
	object->unbound = 0;
    } else {				/* keyword */
	KlKeyword obj = (KlKeyword) object;
	obj->type = KlKeywordType;
	KlIncRef(obj->atom = KlIntern(p_name+1));
	obj->k_val = 0;
	obj->k_count = 0;
    }
    object->p_list = 0;
    return object;
}

KlO
KlAtomEqual(a1, a2)
    KlAtom a1, a2;
{
    if (a2 != a1)
	return NIL;
    else
	return (KlO) a1;
}

KlO
KlDefVar(argc, argv)
    int argc;
    KlO *argv;
{
    KlO doc;
    KlAtom obj;

    switch (argc) {
    case 1:	
    case 2:
	doc = 0;
	break;
    case 3:
	doc = argv[2];
	break;
    default:
	return KlBadNumberOfArguments((char *) argc);
    }
    obj = (KlAtom) argv[0];
    KlMustBeSymbol(obj, 0);
    if (KlIsAnAtom(obj) && obj->c_val == KlUndef) { /* else dont touch */
	KlDecRef(obj->c_val);
	KlIncRef(obj->c_val = (argc == 1 ? NIL : KlSend_eval(argv[1])));
    }
    if (doc)
	KlDocumentationSet(obj, doc);
    return (KlO) obj;
}

/************************\
* 			 *
*  hash table routines 	 *
* 			 *
\************************/

/*
 * Hash function definition:
 * HASH_FUNCTION: hash function, hash = hashcode, hp = pointer on char,
 *				 hash2 = temporary for hashcode.
 * INITIAL_TABLE_SIZE in slots
 * HASH_TABLE_GROWS how hash table grows.
 */

/* KLONE hash function for symbols
 */
#define HASH_FUNCTION_ORIG 	  hash = (hash << 3) + (hash >> 28) + *hp++;
/* int l, inc; unsigned int hash; unsigned char *p, *last; */
#define HASH_FUNCTION(l, p, last, hash, inc) \
    if ((l) > 8) for (hash = 0, inc = ((l)>>3) ; p < last;p += inc) \
                     hash = (hash << 3) + (hash >> 28) + *p; \
    else for (hash = 0; p < last;) hash = (hash << 3) + (hash >> 28) + *p++
#define INITIAL_HASH_SIZE 2017
#define HASH_TABLE_GROWS  KlHashTableSize = KlHashTableSize * 2 + 1;

/* end of hash functions */

/*
 * The hash table is used to store atoms via their P_NAME:
 *
 * P_NAME --hash--> ATOM |--p_name--> "foo"
 *			 |--c_val--> value of the atom (result of eval)
 *
 * if c_val is KlUndef, symbol was undefined.
 * Parsing replaces p_names with ATOMS.
 */

int KlHashTableSize;			/* only used externally for debug */
KlAtom *KlAtomHashTable;

static int HashTableLimit;		/* private */
static int HashTableUsed;
/*
 * KlHashSlot gives the slot (pointer to KlAtom) of a name
 * (slot points to NULL if it is not defined)
 */

KlAtom *
KlHashSlotBytes(l, s)
    int l;
    char *s;
{
    register unsigned int hash;
    register int inc;
    KlAtom *p;
    register unsigned char *hp = (unsigned char *) s, *last = hp + l;
    char *ns;

    HASH_FUNCTION(l, hp, last, hash, inc);
    p = KlAtomHashTable + hash % KlHashTableSize;
    while (*p) {
	ns = (*p)->p_name;
	if (*ns == *s && !bcmp(ns, s, l) && !(ns[l]))
	    break;
	if (--p < KlAtomHashTable)
	    p = KlAtomHashTable + KlHashTableSize - 1;
    }
    return p;
}

KlAtom *
KlHashSlot(s)
    char *s;
{
    return KlHashSlotBytes(strlen(s), s);
}

KlHashTableGrows()
{
    KlAtom *t, *p;
    int i;
    int OldHashTableSize = KlHashTableSize;

    t = KlAtomHashTable;
    HASH_TABLE_GROWS
	HashTableLimit = KlHashTableSize / 3;
    KlAtomHashTable = (KlAtom *) Malloc(KlHashTableSize * sizeof(KlAtom));
    for (p = KlAtomHashTable + KlHashTableSize; p > KlAtomHashTable;)
	*--p = NULL;
    for (i = 0; i < OldHashTableSize; i++)
	if (t[i]) {
	    KlAtom *ps = KlHashSlot(t[i]->p_name);

	    *ps = t[i];
	}
    Free(t);
}

/*****************************************************************************\
* 			       intern & friends                               *
\*****************************************************************************/
/*
 * KlIntern(name)
 * return an KlAtom, which is the one at the slot, if present,
 * or is created if name didn't exist, with c_val KlUndef.
 * This function is called by the parser for each NAME encountered.
 * so that the parsed expression points directly to atoms.
 * The reference count of the atom is set to 1.
 *
 * KlKeywordMake uses the same code later.
 */

KlAtom
KlIntern(tag)
    char *tag;
{
    return KlInternBytes(strlen(tag), tag);
}

/* Faster optimized version of intern, mainly for parsing purposes
 */

KlAtom
KlInternBytes(len, tag)
    int len;				/* length of tag */
    char *tag;
{
    KlAtom *slot = KlHashSlotBytes(len, tag);

    if (!*(slot)) {	/* undefined, make a new one */
	KlAtom object;
	KlAtomCreated = 1;
	/* here is the code of KlAtomMake */
	object = (KlAtom) KlOMake(KlAtomType);

	object->p_name = (char *) Malloc(len + 1);
	bcopy(tag, object->p_name, len);
	object->p_name[len] = '\0';
	if (*tag != ':') {		/* atom */
	    KlIncRef(object->c_val = KlUndef);
	    object->unbound = 0;
	} else {			/* keyword */
	    KlKeyword obj = (KlKeyword) object;
	    obj->type = KlKeywordType;
	    KlIncRef(obj->atom = KlInternBytes(len - 1, tag+1));
	    obj->k_val = 0;
	    obj->k_count = 0;
	}
	object->p_list = 0;
	/* end of KlAtomMake code */

	KlIncRef((*slot = object));
	if (HashTableUsed >= HashTableLimit) {
	    KlAtom newer = *slot;

	    KlHashTableGrows();
	    HashTableUsed++;
	    return newer;
	}
	HashTableUsed++;
    }
    return *slot;
}


/* KlInternKl
 * klone-callable version of KlIntern
 */

KlO
KlInternKl(s)
    KlString s;
{
    KlMustBeString(s, 0);
    return (KlO) KlInternBytes(KlStringLength(s), s->string);
}

/* KlAtomUnbind
 * Removes an atom from the hash table
 */

KlO
KlAtomUnbind(obj)
    KlAtom obj;
{
    int not_already_unbound;

    KlMustBeSymbol(obj, 0);

    if ((obj->type == KlAtomType || obj->type == KlConstantType)
	 && obj->unbound) {
	/* already unbound */
	not_already_unbound = 0;
    } else {
	not_already_unbound = 1;
    }
    /* converts symbol to an atom of value undefined */
    KlAtomInternalsFree(obj);
    obj->type = KlAtomType;
    obj->unbound = 1;
    KlIncRef(obj->c_val = KlUndef);

    /* then this atom will be forgotten as soon as nobody points to it */
    if (not_already_unbound) {
	KlDecRef(obj);
    }

    return NIL;
}

/* must be called before allocating any atom
 */

KlAtomHashTableInit()
{
    KlAtom *p;

    KlHashTableSize = INITIAL_HASH_SIZE;
    HashTableLimit = KlHashTableSize / 3;
    KlAtomHashTable = (KlAtom *) Malloc(KlHashTableSize * sizeof(KlAtom));
    for (p = KlAtomHashTable + KlHashTableSize; p > KlAtomHashTable;)
	*--p = NULL;
}

#ifdef STATS

/*
 * KlHashstats:
 * statistics about the hash table
 */

KlO
KlHashstats()
{
    int out_of_place;

    KlPuts("Statistics about hash table:\n");
    KlPrintf("  %d slots used ", HashTableUsed);
    KlPrintf("out of %d allocated\n", KlHashTableSize);
    out_of_place = KlOutplacedslots();
    KlPrintf("  and %d slots out of place ", out_of_place);
    KlPrintf("(%d.", (out_of_place * 100) / HashTableUsed);
    KlPrintf("%2d %%)\n", ((out_of_place * 10000) / HashTableUsed) % 100);
    return NIL;
}

int
KlOutplacedslots()
{
    KlAtom *slot;
    int n = 0;

    for (slot = KlAtomHashTable; slot < KlAtomHashTable + KlHashTableSize; slot++) {
	if (*slot) {
	    unsigned int hash;
	    unsigned char *hp = (*slot)->p_name;
	    int l = strlen(hp), inc;
	    unsigned char *last = hp + l;
	    char *ns;

	    HASH_FUNCTION(l, hp, last, hash, inc);

	    ns = (*(KlAtomHashTable + hash % KlHashTableSize))->p_name;
	    if (!(ns[0] == (*slot)->p_name[0] &&
		    strcmp(ns, (*slot)->p_name) == 0)) {
		n++;
	    }
	}
    }
    return n;
}

#endif					/* STATS */

/* (do-all-symbols (var [return-form]) body) 
 */

KlO
KlDoAllSymbols(argc, argv)
    int argc;
    KlList *argv;
{
    KlList list, varlist;
    KlO var, result;
    int i;
    int stackptr = KlStackPtr;
    KlAtom *slot;
    KlGCDecls;
    KlDebugStackDecls;

    if (argc < 1)
	return KlBadNumberOfArguments((char *) argc);
    varlist = argv[0];
    KlMustBeList(varlist, 0);
    if (varlist->size < 1 || varlist->size >2) {
	return KlError0(KlE_BAD_DO_SYNTAX);
    }
    var = varlist->list[0];

    KlStackFramePush(1, &var, NIL, NIL);
    KlDebugStackPush(KlSFID_normal, NIL);

    KlGCSet();
    for (slot = KlAtomHashTable;
	 slot < KlAtomHashTable + KlHashTableSize;
	 slot++) {
	if (*slot) {
	    if (((*slot)->p_name[0] > ' ')
		&& ((*slot)->c_val != KlUndef)) {
		KlSend_setq(var, (*slot));
		KlProgn(argc - 1, argv + 1);
		KlGC();
	    }
	}
    }

    if (varlist->size == 2) {
	result = KlSend_eval(varlist->list[1]);
    } else {
	result = NIL;
    }

    KlDebugStackPop();
    KlStackFramePopNormal(stackptr);

    return result;
}


/*
 * KlAtom_eval:
 * evaluating an atom is giving a pointer to its c_val field.
 * an atom returns its value, or calls KlError if undefined
 * (increase ref. of value)
 */

KlO
KlAtomEval(obj)
    KlAtom obj;
{
    if (obj->c_val != KlUndef || obj == (KlAtom) KlUndef) {
	return obj->c_val;
    } else
	return KlError(KlE_UNDEFINED_VARIABLE, obj);
}

/*
 * KlAtomPrint:
 * printing an atom is printing the string in the p_name field.
 */

KlO
KlAtomPrint(obj, stream)
    KlAtom obj;
    KlO stream;
{
    KlSPuts(obj->p_name, stream);		/* perhaps () for NIL? */
    return (KlO) obj;
}

/*
 * KlAtomFree;
 */

KlO
KlAtomFree(obj)
    KlAtom obj;
{
    KlAtom *slot = KlHashSlotBytes(strlen(obj->p_name), obj->p_name);
    KlAtom *next_slot = slot - 1;
    KlAtom atom;

    if (obj == (KlAtom) KlUndef) {	/* we must never free KlUndef! */
	KlIncRef(obj);
	return (KlO) obj;
    }
    if (obj->unbound && obj->c_val != KlUndef) { /* setq-ed since makunbound */
	obj->unbound = 0;
	KlIncRef(obj);			/* save it from deletion */
	return (KlO) obj;
    }
    KlDecRef(obj->c_val);

    *slot = NULL;
    while (atom = *(next_slot = (next_slot < KlAtomHashTable ?
				 KlAtomHashTable + KlHashTableSize - 1
				 : next_slot))) {
	*next_slot = NULL;
	*(KlHashSlotBytes(strlen(atom->p_name), atom->p_name)) = atom;
	next_slot--;
    }
    Free(obj->p_name);
    KlDecRef(obj->p_list);
    Free(obj);

    return (KlO) obj;
}

/* KlAtomInternalsFree
 * utility routine to decrement ref count on internal fields:
 * KlAtom, KlKeyword, KlConstant:   only c_val
 * KlActive:                        nothing
 * KlKloneActive:                    c_val, dummy1, dummy2
 */

KlAtomInternalsFree(obj) 
    KlAtom obj;
{
    if (obj->type != KlActiveType) {
	if (obj->type == KlKloneActiveType) {
	    KlDecRef(((KlKloneActive) obj)->get);
	    KlDecRef(((KlKloneActive) obj)->set);
	} else if ((obj->type == KlAtomType || obj->type == KlConstantType)
		   && obj->unbound) {
	    obj->unbound = 0;
	    KlIncRef(obj);		/* cancels unbounding */
	}
	KlDecRef(obj->c_val);
    }					/* nothing done on actives */
}

/* KlAtomInternalsAdd
 * utility routine to increment ref count on internal fields:
 * KlAtom, KlKeyword, KlConstant:   only c_val
 * KlActive:                        nothing
 * KlKloneActive:                    c_val, dummy1, dummy2
 */

KlAtomInternalsAdd(obj) 
    KlAtom obj;
{
    if (obj->type != KlActiveType) {
	if (obj->type == KlKloneActiveType) {
	    KlIncRef(((KlKloneActive) obj)->get);
	    KlIncRef(((KlKloneActive) obj)->set);
	}
	KlIncRef(obj->c_val);
    }					/* nothing done on actives */
}

/*
 * KlAtomExecute:
 * executes the object in the C_val
 * we just check for infinite loop (otherwise (t) crashes)
 * depth is limited to 10
 */

KlO
KlAtomExecute(obj, list)
    KlAtom obj;
    KlList list;
{
    if (obj->c_val != KlUndef) {
	int depth = 0;
	KlAtom atom = obj;

	while (atom->c_val->type == obj->type) {
	    if (depth++ > 10) {
		goto not_found;
	    } else {
		atom = (KlAtom) atom->c_val;
	    }
	}
	return KlSend_execute(obj->c_val, list);
    } 

 not_found:
    {
	KlO func = KlError(KlE_UNDEFINED_FUNCTION, obj);
	return KlSend_execute(func, list);
    }
}
/*
 * KlAtomApply:
 * executes the object in the C_val
 * we just check for infinite loop (otherwise (t) crashes)
 * depth is limited to 10
 */

KlO
KlAtomApply(obj, list)
    KlAtom obj;
    KlList list;
{
    if (obj->c_val != KlUndef) {
	int depth = 0;
	KlAtom atom = obj;

	while (atom->c_val->type == obj->type) {
	    if (depth++ > 10) {
		KlO func = KlError(KlE_UNDEFINED_FUNCTION, obj);
		return KlSend_apply(func, list);
	    } else {
		atom = (KlAtom) atom->c_val;
	    }
	}
	return KlSend_apply(obj->c_val, list);
    } else {
	KlO func = KlError(KlE_UNDEFINED_FUNCTION, obj);
	return KlSend_apply(func, list);
    }
}

/*
 * KlAtomSet
 * the normal setq routine
 */

KlO
KlAtomSetq(atom, value)
    KlAtom atom;
    KlO value;
{
    KlDecRefNonNull(atom->c_val);
    KlIncRef(atom->c_val = value);
    return value;
}

/* KlAtomCoerce
 * method to creat an atom
 */

/*ARGSUSED*/
KlO
KlAtomCoerce(totype, obj)
    KlType totype;
    KlO obj;
{
    if (KlIsAString(obj)) {
	return (KlO) KlIntern(((KlString) obj)->string);
    } else if (KlIsAKeyword(obj)) {
	return (KlO) ((KlKeyword) obj)->atom;
    } else {
	return 0;
    }    
}

/* KlKeywordCoerce
 */

/*ARGSUSED*/
KlO
KlKeywordCoerce(totype, obj)
    KlType totype;
    KlO obj;
{
    if (!KlIsASymbol(obj)) 
	obj = KlCoerce(obj, KlAtomType);
    if (obj)
	return (KlO) KlKeywordMakeFromAtom(obj);
    else
	return 0;
}

KlO
KlAtomLengthKl(obj)
    KlAtom obj;
{
    return (KlO) KlNumberMake(strlen(obj->p_name));
}


KlO
KlAtomNth(obj, i, value)
    KlAtom obj;
    UInt i;
    KlNumber value;
{
    if ((i >= strlen(obj->p_name)) || value) {
	return NIL;
    } else {
	return (KlO) KlNumberMake(obj->p_name[i]);
    }
}


/**************************************************************** misc utils */

/* KlInternDotPrefixed
 * interns a.b
 */

KlAtom
KlInternDotPrefixed(a, b)
    char *a;
    char *b;
{
    KlAtom result;
    char *dottedname = (char *) Malloc(strlen(a) + 2 + strlen(b));

    strcpy(stpcpy(stpcpy(dottedname,
			 a),
		  "."),
	   "b");
    result = KlIntern(dottedname);
    Free(dottedname);
    return result;
}  

KlMustBeAtomOrString(variable, pos)
    KlAtom *variable;
    int pos;
{
    if (!KlIsASymbol(*variable)) {
	if (KlIsAString(*variable)) {
	    *variable = (KlAtom) KlInternKl(*variable);
	} else {
	    KlMustBeAtom(((KlO) *variable), pos);
	}
    }
}

/*****************************************************************************\
* 				   keywords                                   *
\*****************************************************************************/

KlKeyword
KlKeywordMakeFromAtom(atom)
    KlAtom atom;
{
    char *tmp = (char *) Malloc(strlen(atom->p_name) + 2);
    KlKeyword key;

    tmp[0] = ':';
    strcpy(tmp + 1, atom->p_name);
    key = KlKeywordMake(tmp);
    Free(tmp);
    return key;
}


KlO
KlKeywordHash(k)
    KlKeyword k;
{
    return (KlO) k->atom;		/* so that (= :foo 'foo) works */
}

/*****************************************************************************\
* 				  constants                                   *
\*****************************************************************************/

/* KlConstantMake
 */

KlConstant
KlConstantMake(name, value)
    char *name;
    KlO value;
{
    KlConstant obj = (KlConstant) KlIntern(name);

    obj->type = KlConstantType;
    KlIncRef(obj->c_val = value);
    return obj;
}

/* klone-callable defconstant
 */

KlO
KlDefConstant(argc, argv)
    int argc;
    KlO *argv;
{
    KlO doc;
    KlConstant obj;

    switch (argc) {
    case 2:
	doc = 0;
	break;
    case 3:
	doc = argv[2];
	break;
    default:
	return KlBadNumberOfArguments((char *) argc);
    }
    KlMustBeSymbol(argv[0], 0);
    if (KlIsAnAtom(argv[0]) || KlIsAConstant(argv[0])) {
	KlDecRef(((KlAtom) argv[0])->c_val);
    }
    obj = KlConstantMake(((KlAtom) argv[0])->p_name, KlSend_eval(argv[1]));
    if (doc)
	KlDocumentationSet(obj, doc);
    return obj->c_val;
}

KlO
KlRedefConstant(obj, value)
    KlAtom obj;
    KlO value;
{
    if (KlIsAConstant(obj) || KlIsAnAtom(obj)) {
	return KlAtomSetq(obj, value);
    } else {
	return KlSend_setq(obj, value);
    }
}

/*****************************************************************************\
* 			     Current package hack                             *
\*****************************************************************************/

/* KlPacknameMake
 * takes a symbol to be expanded in current package, and interns it
 * now supposes we must strip first char of name
 */

KlAtom
KlPacknameMake(len, name)
    int len;
    char *name;
{
    static char *KlPacknameMakeTempstring;
    static int KlPacknameMakeTempstringLength;
    int expansed_name_length;

    if (!KlModStringLength(KlPackage)) { /* *package* () or "" ==> main */
	return KlInternBytes(len - 1, name + 1); /* skip % char */
    }

    expansed_name_length = KlModStringLength(KlPackage) + len;
    if (expansed_name_length > KlPacknameMakeTempstringLength) {
	KlPacknameMakeTempstringLength = expansed_name_length;
	KlPacknameMakeTempstring = (char *)
	    Realloc(KlPacknameMakeTempstring, expansed_name_length + 1);
    }
    strcpy(stpcpy(stpcpy(KlPacknameMakeTempstring,
			 KlPackage->string),
		  ":"),
	   name + 1);

    return KlInternBytes(expansed_name_length, KlPacknameMakeTempstring);
}

#ifdef MLEAK

/* For malloc debugging:
 * gives the atoms with prefix prefix successivly (or NULL on end);
 * re-initialise with a '\0' prefix
 */

KlAtom
KlAtomFindNextPrefixedAtom(prefix)
    char prefix;
{
    KlAtom *slot;
    static KlAtom *slot0;

    if (prefix == '\0') {
	slot0 = KlAtomHashTable;
    } else {
	for (slot = slot0; slot < KlAtomHashTable + KlHashTableSize; slot++)
	    if ((*slot)
		&& ((*slot)->p_name[0] == prefix)
		&& ((*slot)->c_val != KlUndef)
		&& ((*slot)->c_val)	/* skip null counts */
		) {
		slot0 = slot + 1;
		return (*slot);
	    }
    }
    return NULL;
}

#endif					/* MLEAK */

KlO
KlStringHash(obj)
    KlString obj;
{
    register unsigned int hash, inc;
    unsigned char *hp = (unsigned char *) obj->string;
    unsigned char *last = hp + obj->size;
    HASH_FUNCTION(obj->size, hp, last, hash, inc);
    return (KlO) hash;
}

      
/*****************************************************************************\
*                                                                             *
* 				Active Values                                 *
*                                                                             *
\*****************************************************************************/

/*
 * Actives stand for "active-values", i.e. variables triggering a function
 * when evaluated and another when set. Used like atoms to decrease number
 * of parentheses
 */

/*
 * KlActiveMake
 * High level function callable from C. Makes a active with a string and a
 * C active
 * warning: do a KlIncRef on data if it is a klone object!!!
 */

KlActive
KlActiveMake(name, get, set, data)
    char *name;
    KlMethod get, set;
    KlO data;
{
    KlActive obj;

    KlAtomCreated = 0;
    obj = (KlActive) KlIntern(name);
    KlAtomInternalsFree(obj);

    obj->type = KlActiveType;
    obj->get = get;
    obj->set = set;
    obj->data = data;
    return obj;
}

/*
 * Evaluating an active executes the GET function with contents of the data
 * field as parameter
 */

KlO
KlActiveEval(obj)
    KlActive obj;
{
    if (KlActiveGet(obj))
	return (KlO) CFAPPLY(KlActiveGet(obj), (KlActiveData(obj)));
    else
	return NIL;
}

/*
 * Setting an active executes the SET function with arguments the value to be
 * set to, and the contents of the data field
 *
 * WARNING: if your function *(KlActiveSet(obj)) calls KlError, the call to
 * KlError will return! (with NIL value), so be careful to return from your
 * function just after calling KlError...
 */

KlO
KlActiveSetq(obj, value)
    KlActive obj;
    KlO value;
{
    if (KlActiveSet(obj))
	return (KlO) CFAPPLY(KlActiveSet(obj), (value, KlActiveData(obj)));
    else
	return KlError(KlE_CANNOT_SET, obj);
}

/*
 * KlActiveExecute:
 * 	with 1 arg, executes set
 * 	with 0 arg, executes get
 */

KlO
KlActiveExecute(obj, list)
    KlActive obj;
    KlList list;
{
    if ((list->size == 1) && KlActiveGet(obj))	/* GET */
	return CFAPPLY(KlActiveGet(obj), (KlActiveData(obj)));
    else if (KlActiveSet(obj))		/* SET */
	return CFAPPLY(KlActiveSet(obj),
	    (KlSend_eval((list->list)[1]), KlActiveData(obj)));
    else
	return NIL;
}

/*
 * KlActiveApply
 * same as execute, but do not eval argument
 */

KlO
KlActiveApply(obj, list)
    KlActive obj;
    KlList list;
{
    if ((list->size == 1) && KlActiveGet(obj)) /* GET */
	return CFAPPLY(KlActiveGet(obj), (KlActiveData(obj)));
    else if (KlActiveSet(obj))		/* SET */
	return CFAPPLY(KlActiveSet(obj), ((list->list)[1], KlActiveData(obj)));
    else
	return NIL;
}

/*****************************************************************************\
* 				 klone actives                                 *
\*****************************************************************************/

KlO
KlKloneActiveMake(name, get, set, data)
    KlAtom name;
    KlO get;
    KlO set;
    KlO data;
{
    KlKloneActive obj;

    KlMustBeAtomOrString(&name, 0);
    
    KlAtomCreated = 0;
    obj = (KlKloneActive) name;
    KlAtomInternalsFree(obj);

    obj->type = KlKloneActiveType;
    KlIncRef(obj->get = KlSend_eval(get));
    KlIncRef(obj->set = KlSend_eval(set));
    KlIncRef(obj->data = KlSend_eval(data));
    return (KlO) obj;
}

/* KlKloneActiveEval
 * apply the call (obj->get obj) and return the result
 * return the result or the data field if obj->get is NIL.
 */

KlO
KlKloneActiveEval(obj)
    KlKloneActive obj;
{
    if (KlTrueP(obj->get)) {
	return KlSend_apply(obj->get, KlListPairMake(obj->get, obj));
    } else {
	return obj->data;
    }
}

/* KlKloneActiveSetq
 * apply the call (obj->set obj value) and return the result
 * if obj->set is NIL, sets the data field to value
 */

KlO
KlKloneActiveSetq(obj, value)
    KlKloneActive obj;
    KlO value;
{
    if (KlTrueP(obj->set)) {
	KlList list = KlListTripletMake(obj->set, obj, value);
	return KlSend_apply(obj->set, list);
    } else {
	KlDecRef(obj->data);
	KlIncRef(obj->data = value);
	return value;
    }
}

/* KlKloneActiveExecute:
 * 	with 1 arg, executes set
 * 	with 0 arg, executes get
 */

KlO
KlKloneActiveExecute(obj, list)
    KlKloneActive obj;
    KlList list;
{
    if (list->size == 1) {		/* GET */
	return KlKloneActiveEval(obj);
    } else {		/* SET */
	return KlKloneActiveSetq(obj, KlSend_eval(list->list[1]));
    }
}

/*
 * KlActiveApply
 * same as execute, but do not eval argument
 */

KlO
KlKloneActiveApply(obj, list)
    KlKloneActive obj;
    KlList list;
{
    if (list->size == 1) {		/* GET */
	return KlKloneActiveEval(obj);
    } else {		/* SET */
	return KlKloneActiveSetq(obj, list->list[1]);
    }
}

/* Put & Get interface to the data field (named 'slot) */

KlO
KlKloneActiveGet(obj, key, def)
    KlKloneActive obj;
    KlAtom key;
    KlO def;
{
    if (key == KlA_slot) {
	return obj->data;
    } else if (key == KlA_get) {
	return obj->get;
    } else if (key == KlA_put) {
	return obj->set;
    } else {
	return KlExecuteGetDefault(obj, key, def);
    }
}

KlO
KlKloneActivePut(obj, key, val)
    KlKloneActive obj;
    KlAtom key;
    KlO val;
{
    if (key == KlA_slot) {
	KlSetField(&(obj->data), val);
    } else if (key == KlA_get) {
	KlSetField(&(obj->get), val);
    } else if (key == KlA_put) {
	KlSetField(&(obj->set), val);
    } else {
	return KlError2(KlE_NO_ELEMENT, key, KlTypeName(KlKloneActiveType));
    }
    return val;
}

/*****************************************************************************\
* 				   pointers                                   *
\*****************************************************************************/
/* A pointer is an active with methods to automatically update a C memory
 * location
 * It is made by the C call:
 * KlActiveMake(<name>, KlActivePointerToIntGet, KlActivePointerToIntSet,
 *		  <pointer-to-integer-to-update>);
 */

/************************* the function to be used to refer to an integer */

/* the function to be used to get value of pointer as a klone number
 */

KlO
KlActivePointerToIntGet(data)
    int *data;
{
    return (KlO) KlNumberMake(*data);
}

/* the function to be used to set value of pointer to a klone number
 */

KlO
KlActivePointerToIntSet(num, data)
    KlNumber num;
    int *data;
{
    if (KlIsANumber(num)) {
	*data = num->number;
    } else if (KlFalseP(num)) {
	*data = 0;
    } else if (num == (KlNumber) TRU) {
	*data = 1;
    } else {
	KlActivePointerToIntSet(KlCoerce(num, KlNumbersType), data);
    }
    return (KlO) num;
}

KlActive
KlActivePointerToIntMake(name, ptr)
    char *name;
    int *ptr;
{
    return KlActiveMake(name, KlActivePointerToIntGet,
	KlActivePointerToIntSet, ptr);
}

/************************** the function to be used to refer to a boolean */

KlO
KlActivePointerToBooleanGet(data)
    int *data;
{
    return (*data ? TRU : NIL);
}

KlO
KlActivePointerToBooleanSet(num, data)
    KlO num;
    int *data;
{
    *data = KlTrueP(num) ? 1 : 0;
    return num;
}

KlActive
KlActivePointerToBooleanMake(name, ptr)
    char *name;
    int *ptr;
{
    return KlActiveMake(name, KlActivePointerToBooleanGet,
	KlActivePointerToBooleanSet, ptr);
}

/********************** the function to be used to refer to a KlO pointer */

KlO
KlActivePointerToObjGet(data)
    KlO *data;
{
    return (*data ? *data : NIL);
}

KlO
KlActivePointerToObjSet(obj, data)
    KlO obj;
    KlO *data;
{
    *data = obj;
    return obj;
}

KlActive
KlActivePointerToObjMake(name, ptr)
    char *name;
    KlO *ptr;
{
    return KlActiveMake(name, KlActivePointerToObjGet,
	KlActivePointerToObjSet, ptr);
}

/************************** the function to be used to refer to a KlString */

KlO
KlActivePointerToKloneStringGet(data)
    KlString *data;
{
    return (KlO) *data;
}

KlO
KlActivePointerToKloneStringSet(obj, data)
    KlString obj;
    KlString *data;
{
    if (KlTrueP(obj)) {
	KlMustBeString(obj, 1);
	if (KlIsASymbol(obj)) {		/* use a real string instead of atom */
	    obj = KlStringMake(obj->string);
	}
    } else {
	obj = KlNilString;
    }
    KlDecRefNonNull(*data);
    KlIncRef(*data = obj);
    return (KlO) obj;
}

KlActive
KlActivePointerToKloneStringMake(name, ptr)
    char *name;
    KlString *ptr;
{
    return KlActiveMake(name, KlActivePointerToKloneStringGet,
	KlActivePointerToKloneStringSet, ptr);
}

/*****************************************************************************\
* 				 Symbol Slots                                 *
\*****************************************************************************/
/* Symbol slots are place holders, slots in the hash table pointing to the 
 * actual symbols themselves, which can be atoms, active, constants, keywords,
 * and klone-actives
 * 
 * This system only works because all symbols occupy the same memory size
 */

KlSymbolSlot
KlSymbolSlotMake(name)
    KlAtom name;
{
    KlSymbolSlot object;

    KlMustBeAtomOrString(&name, 0);

    object = (KlSymbolSlot) KlOMake(KlSymbolSlotType);

    /* we must incref it so it is not freed by makunbound */
    KlIncRef(object->symbol = name);

    return object;
}

KlO
KlSymbolSlotCoerce(totype, obj)
    KlType totype;
    KlO obj;
{
    if (KlIsAString(obj)) {
	return (KlO) KlSymbolSlotMake(obj);
    } else {
	return 0;
    }
}

KlO
KlSymbolSlotFree(object)
    KlSymbolSlot object;
{
    KlDecRef(object->symbol);
    Free(object);
    return (KlO) object;
}

/* returns a snapshot (inactive copy) of what was at the slot */
KlO 
KlSymbolSlotEval(symbslot)
    KlSymbolSlot symbslot;
{
    /* copy it */
    KlO snapshot = (KlO) Malloc(sizeof(struct _KlAtom));
    bcopy(symbslot->symbol, snapshot, sizeof(struct _KlAtom));

    /* reset reference count of the copy since nobody points to it now */
    KlResetRef(snapshot);

    return snapshot;
}


/* replace physically an atom by another in the global symbol hashtable */
/* WARNING: this is of course inherently dangerous!!! */ 

KlO 
KlSymbolSlotSetq(symbslot, value)
    KlSymbolSlot symbslot;
    KlAtom value;
{
    /* save infos */
    KlAtom atom = symbslot->symbol;
    int old_refcount = KlRef(atom);

    if (value == (KlAtom) NIL)
	value = (KlAtom) KlUndef;	/* setting to NIL undefines symbol */
    KlMustBeSymbol(value, 1);

    /* properly clean internal infos of the symbol */
    KlAtomInternalsFree(atom);
    /* then copy infos and set refcount */
    atom->c_val = value->c_val;
    atom->dummy = value->dummy;
    atom->unbound = value->unbound;
    KlAtomInternalsAdd(atom);

    KlRef(atom) = old_refcount;
    atom->type = value->type;
    return (KlO) value;
}

/*****************************************************************************\
* 				  TYPE INIT                                   *
\*****************************************************************************/
/* WARNING: do NOT declare funcs here, via KlDeclareSubr since the functions
 * are not yet initialized!!!
 */

KlAtomInit()
{
    KlAtomHashTableInit();

    /* now some fancy bootstraping code: pre-alloc undefined value for atoms 
     * since KlAtomType and KlUndef are mutually dependent
     */
    {
	char *tag = "*undefined*";
	struct _KlO dummy;			/* dummy value */
	KlAtom *slot = KlHashSlot(tag);

	dummy.reference_count = 0;
	KlAtomBucket = KlMallocBucketOfSize(sizeof(struct _KlAtom));
	KlIncRef((*slot = KlAtomMake(tag, &dummy)));
	HashTableUsed++;

	KlUndef = (KlO) *slot;
	KlIncRef(((KlAtom)KlUndef)->c_val = KlUndef);
    }

    KlDeclareType(&KlAtomType, "Atom", sizeof(struct _KlAtom));

    /* then complete KlUndef now that KlAtomType is defined */
    KlUndef->type = KlAtomType;

    KlDeclareIsTrait(KlAtomType, KlTrait_symbol);
    KlDeclareTrait(KlAtomType, KlTrait_string);
    KlDeclareTrait(KlAtomType, KlTrait_hasheq);

    KlDeclareMethod1(KlAtomType, KlSelEval, KlAtomEval);
    KlDeclareMethod1(KlAtomType, KlSelPrint, KlAtomPrint);
    KlDeclareMethod1(KlAtomType, KlSelFree, KlAtomFree);
    KlDeclareMethod1(KlAtomType, KlSelExecute, KlAtomExecute);
    KlDeclareMethod1(KlAtomType, KlSelApply, KlAtomApply);
    KlDeclareMethod1(KlAtomType, KlSelEqual, KlAtomEqual);
    KlDeclareMethod1(KlAtomType, KlSelSetq, KlAtomSetq);
    KlDeclareMethod1(KlAtomType, KlSelLength, KlAtomLengthKl);

					/* string inheritance */
    KlDeclareMethod1(KlAtomType, KlSelNth, KlAtomNth);
    KlDeclareMethod1(KlAtomType, KlSelAdd, KlStringAdd);
    KlDeclareMethod1(KlAtomType, KlSelGet, KlStringGet);

    KlDeclareSubType(&KlConstantType, "Constant", KlAtomType,
		     sizeof(struct _KlConstant));

    KlDeclareMethod1(KlConstantType, KlSelSetq, KlErrorCannotSet);

    KlDeclareSubType(&KlKeywordType, "Keyword", KlAtomType,
		     sizeof(struct _KlKeyword));

    KlDeclareMethod1(KlKeywordType, KlSelEval, KlQuote);
    KlDeclareMethodUndefined(KlKeywordType, KlSelExecute);
    KlDeclareMethod1(KlKeywordType, KlSelSetq, KlErrorCannotSet);
    KlDeclareMethod1(KlKeywordType, KlSelHash, KlKeywordHash);
    KlDeclareSubType(&KlActiveType, "Active", KlAtomType,
		     sizeof(struct _KlActive));

    KlDeclareMethod1(KlActiveType, KlSelEval, KlActiveEval);
    KlDeclareMethod1(KlActiveType, KlSelExecute, KlActiveExecute);
    KlDeclareMethod1(KlActiveType, KlSelApply, KlActiveApply);
    KlDeclareMethod1(KlActiveType, KlSelSetq, KlActiveSetq);

    KlDeclareSubType(&KlKloneActiveType, "ExActive", KlAtomType,
		     sizeof(struct _KlKloneActive));

    KlDeclareMethod1(KlKloneActiveType, KlSelEval, KlKloneActiveEval);
    KlDeclareMethod1(KlKloneActiveType, KlSelExecute, KlKloneActiveExecute);
    KlDeclareMethod1(KlKloneActiveType, KlSelApply, KlKloneActiveApply);
    KlDeclareMethod1(KlKloneActiveType, KlSelSetq, KlKloneActiveSetq);
    KlDeclareMethod1(KlKloneActiveType, KlSelGet, KlKloneActiveGet);
    KlDeclareMethod1(KlKloneActiveType, KlSelPut, KlKloneActivePut);

    KlDeclareType(&KlSymbolSlotType, "SymbolSlot",
		  sizeof(struct _KlSymbolSlot));

    KlDeclareMethod1(KlSymbolSlotType, KlSelEval, KlSymbolSlotEval);
    KlDeclareMethod1(KlSymbolSlotType, KlSelSetq, KlSymbolSlotSetq);
    
    KlA_star_package = KlActivePointerToKloneStringMake("*package*", &KlPackage);
    KlA_nil_symbol = KlIntern("*()*");
    KlA_print_binary = KlIntern("*print-binary*");
    KlA_load_path = KlIntern("*load-pathname*");
    KlA_load_ext = KlIntern("*load-extensions*");
    KlA_EOF = KlIntern("EOF");
    KlA_EOP = KlIntern("EOP");
    KlA_ERROR = KlIntern("ERROR");
    KlA_ALL = KlIntern("ALL");
    KlA_error_handlers = KlIntern("*error-handlers*");
    KlA_error_handlers_orig = KlIntern("*error-handlers-orig*");
    KlA_error_correctors = KlIntern("*error-correctors*");
    KlA_smartloader = KlIntern("smartloader");
    KlA_is_in_error_handler = KlIntern("*is-in-error-handler*");
    KlA_quote = KlIntern("'");
    KlA_backquote = KlIntern("`");
    KlA_unquote = KlIntern(",");
    KlA_unquotesplicing = KlIntern(",@");
    KlA_print = KlIntern("print");
    KlA_Moptional = KlIntern("&optional");
    KlA_Mrest = KlIntern("&rest");
    KlA_Mkey = KlIntern("&key");
    KlA_Maux = KlIntern("&aux");
    KlA_Mwhole = KlIntern("&whole");
    KlA_Mallow_other_keys = KlIntern("&allow-other-keys");
    KlA_Class = KlIntern("Class");
    KlA_Object = KlIntern("Object");
    KlA_continue = KlIntern("continue");
    KlA_init = KlIntern("init");
    KlA_free = KlIntern("free");
    KlA_execute = KlIntern("execute");
    KlA_nth = KlIntern("nth");
    KlA_hash = KlIntern("hash");
    KlA_class = KlIntern("class");
    KlA_object = KlIntern("object");
    KlA_selector = KlIntern("selector");
    KlA_slot = KlIntern("slot");
    KlA_name = KlIntern("name");
    KlA_body = KlIntern("body");
    KlA_correctable = KlIntern("correctable");

    KlK_direction = KlKeywordMake(":direction");
    KlK_if_exists = KlKeywordMake(":if-exists");
    KlK_type = KlKeywordMake(":type");
    KlK_buffered = KlKeywordMake(":buffered");
    KlK_output = KlKeywordMake(":output");
    KlK_input = KlKeywordMake(":input");
    KlK_error = KlKeywordMake(":error");
    KlK_io = KlKeywordMake(":io");
    KlK_overwrite = KlKeywordMake(":overwrite");
    KlK_string = KlKeywordMake(":string");
    KlK_supersede = KlKeywordMake(":supersede");
    KlK_append = KlKeywordMake(":append");
    KlK_initial_element = KlKeywordMake(":initial-element");
    KlK_real = KlKeywordMake(":real");
    KlK_user = KlKeywordMake(":user");
    KlK_sys = KlKeywordMake(":sys");
    KlK_cpu = KlKeywordMake(":cpu");
    KlK_initform = KlKeywordMake(":initform");
    KlK_documentation = KlKeywordMake(":documentation");
    KlK_nil = KlKeywordMake(":nil");
    KlK_true = KlKeywordMake(":true");
    KlK_writer = KlKeywordMake(":writer");
    KlK_metaclass = KlKeywordMake(":metaclass");
    KlK_name = KlKeywordMake(":name");
    KlK_father = KlKeywordMake(":father");
    KlK_son = KlKeywordMake(":son");
    KlK_brother = KlKeywordMake(":brother");
    KlK_methods = KlKeywordMake(":methods");
    KlK_slots = KlKeywordMake(":slots");
    KlK_STAR_inits = KlKeywordMake("::*inits");
    KlK_exo_hooks = KlKeywordMake("::*exo-hooks");
    KlK_file = KlKeywordMake(":file");
    KlK_blocking = KlKeywordMake(":blocking");
    KlK_timeout = KlKeywordMake(":timeout");
    KlK_nohup = KlKeywordMake(":nohup");
    KlK_if_does_not_exist = KlKeywordMake(":if-does-not-exist");

    KlDeclareSymbol(KlO, TRU, "t", TRU);
    ((KlAtom) TRU)->type = KlConstantType;

    {					/* init KlTypeType name */
	KlAtom typename = (KlAtom) KlConstantMake("Type", KlTypeType);

	KlTypeNameSet(KlTypeType, typename);
	typename = (KlAtom) KlConstantMake("Any", KlAnyType);
	KlTypeNameSet(KlAnyType, typename);
    }
    /* put the type into all type names created before KlConstantType */
    KlTypeName(KlAtomType)->type = KlConstantType;


    KlValidKeywords = (KlKeyword *) Malloc(KLSO * KLMAX_KEYWORDS);

}
