#include "emall.h"

/**************************************************************** Prototypes */


/******************************************************************** Bodies */


/* Call KlError. */
int
EmCommErrorKl(int errCode)
{
    KlAtom klErr;

    klErr = KlErrorCodeMakeS(USErrCodeToErrString(errCode));
    KlError0(klErr);
    return errCode;
}

/* Init the server.
 *
 * Must be called before any call to other function of this lib.
 *
 * Return: NIL
 *
 * Warning: This function set the signal handler for SIG_PIPE to SIG_IGN
 *
 * Wrapper: for int USInit(int portNum, char *serverName);
 */
KlO
EmInitCommServKl(KlO portNum, KlO serverName)
{
    KlMustBeIntOrConstInt(portNum, 0);
    KlMustBeStringOrConstString(serverName, 1);
    EmCheckError(USInit(KlNumToInt(portNum), KlStringToCharPtr(serverName)));
    return NIL;
}

/* Close the line (id).
 *
 * Return: NIL
 *
 * Wrapper: for int USCloseLine(int id);
 */
KlO
EmCloseLineKl(KlO id /* The line */ )
{
    KlMustBeId(id, 0);
    EmCheckError(USCloseLine(EmIdToCommId(KlNumToInt(id))));
    return NIL;
}

/* Check if new applications are requesting for lines.
 * Accept the first one.
 *
 * Return: return the id of the first line or NIL if none
 *
 * Warning: This function must be called periodicaly to insure that connection
 * requests aren't ignored.
 *
 * Wrapper: for int USAcceptNewLine(int *pId, int *pGotLine);
 */
KlO
EmAcceptNewLineKl(void)
{
    KlO ro;
    int id, gotLine;

    EmCheckError(USAcceptNewLine(&id, &gotLine));
    if (gotLine) {
	ro = (KlO) KlNumberMake(CommIdToEmId(id));
	return ro;
    }
    else
	return NIL;
}


/* Wait until new applications are requesting for lines.
 * Accept the first one and return its id
 *
 * Warning: this function is blocking
 *
 * Wrapper: for int USWaitForNewLine(int *pId, int *pGotLine);
 */
KlO
EmWaitForNewLineKl(void)
{
    KlO ro;
    int id, gotLine;

    EmCheckError(USWaitForNewLine(&id));
    ro = (KlO) KlNumberMake(CommIdToEmId(id));
    return ro;
}

/* Send a message (on at least store it in a buffer).
 *
 * Return: NIL
 *
 * Wrapper: for int USSend(int id, void *msg, int len);
 */
KlO
EmSendKl(KlO id, KlO msg)
{
    KlMustBeId(id, 0);
    KlMustBeStringOrConstString(msg, 1);
    EmCheckError(USSend(EmIdToCommId(KlNumToInt(id)), KlStringToCharPtr(msg), strlen(KlStringToCharPtr(msg))));
    return NIL;
}

/* Send all messages which may have been buffered by server line (id) (if any)
 * (ie flush actual packet (if needed)).
 *
 * Return: NIL
 *
 * Wrapper: for int USFlush(int id);
 */
KlO
EmFlushKl(KlO id)
{
    KlMustBeId(id, 0);
    EmCheckError(USFlush(EmIdToCommId(KlNumToInt(id))));
    return NIL;
}

/* Flush messages waiting to be send for all clients.
 *
 * Return: NIL
 *
 * Wrapper for :int USFlushAllClients(void);
 */
KlO
EmFlushAllClientsKl(void)
{
    EmCheckError(USFlushAllClients());
    return NIL;
}

/* How many messages are waiting for a read.
 *
 * Return imediatly if messages are already queued.
 * If no message queued, flush buffer and wait for roundtrip
 * (ie be sure no message are kept in other line end buffer).
 *
 * Return: number of waiting mesg
 *
 * Warning: may block a bit if correspondant is too busy.
 *
 * Wrapper: for int USPendingMsg(int id, int *pNbMsg);
 */
KlO
EmPendingMsgKl(KlO id)
{
    int nbMsg;

    KlMustBeId(id, 0);
    EmCheckError(USPendingMsg(EmIdToCommId(KlNumToInt(id)), &nbMsg));
    KlRetNilForZero(nbMsg);
}

/*
 * How many messages are waiting for a read
 *
 * Checks for waiting packets abd return
 * nether blocks
 *
 * Return: number of waiting mesg.
 *
 * Wrapper: for int USCheckPendingMsg(int id, int *pNbMsg);
 */
KlO
EmCheckPendingMsgKl(KlO id)
{
    int nbMsg;

    KlMustBeId(id, 0);
    EmCheckError(USCheckPendingMsg(EmIdToCommId(KlNumToInt(id)), &nbMsg));
    KlRetNilForZero(nbMsg);
}

/* Checks if there is something waiting for a read on all connected lines.
 *
 * Do the check as fast as possible (only one select)
 * nether blocks
 *
 * Wrapper: for int USCheckAllLines(int *pIsMsg);
 */
KlO
EmCheckAllLinesKl(void)
{
    int isMsg;

    EmCheckError(USCheckAllLines(&isMsg));
    KlRetNilForZero(isMsg);
}



/* Get next message.
 *
 * If no message in waiting queue;
 * Flush this line-end
 * Make periodics roundtrip to insure no messages are in buffer at the
 * other line-end
 *
 * Return: got message.
 *
 * Warning: It's a blocking call.
 *
 * Wrapper: for int USGetNextMsg(int id, char **ppMsg, int *pLen);
 */
KlO
EmGetNextMsgKl(KlO id)
{
    KlO ro;
    char *pMsg;
    int msgLen;

    KlMustBeId(id, 0);
    /* ?? TODO must return size too!!! */
    EmCheckError(USGetNextMsg(EmIdToCommId(KlNumToInt(id)), &pMsg, &msgLen));
    ro = (KlO) KlStringMakeNoCopy(msgLen, pMsg);
    return ro;
}

/* Peek next message (message is not removed from queue).
 *
 * If no message in waiting queue;
 * Flush this line-end
 * Make periodics roundtrip to insure no messages are in buffer at the
 * other line-end
 *
 * Return: NIL or message.
 *
 * Warning: It's a blocking call.
 *
 * Wrapper: for int USPeekNextMsg(int id, char **ppMsg, int *pLen);
 */
KlO
EmPeekNextMsgKl(KlO id)
{
    KlO ro;
    char *pMsg;
    int msgLen;

    KlMustBeId(id, 0);
    /* ?? TODO must return size too!!! */
    EmCheckError(USPeekNextMsg(EmIdToCommId(KlNumToInt(id)), &pMsg, &msgLen));
    if (msgLen) {
	ro = (KlO) KlStringMakeFromBytes(msgLen, pMsg);
	return ro;
    }
    else
	return NIL;
}

/* Return (and remove) first matching message from queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search).
 *
 * Return: got message or NIL.
 *
 * Wrapper: for int USGetFirstMatchingMsg(int id, char *head, char **ppMsg, int *pLen);
 */
KlO
EmGetFirstMatchingMsgKl(KlO id,
			KlO head	/* String which must be the head of
			    matching message */ )
{
    KlO ro;
    char *pMsg;
    int msgLen;

    KlMustBeId(id, 0);
    KlMustBeStringOrConstString(head, 1);

    /* ?? TODO must return size too!!! */
    EmCheckError(USGetFirstMatchingMsg(EmIdToCommId(KlNumToInt(id)), KlStringToCharPtr(head), &pMsg, &msgLen));
    if (msgLen) {
	ro = (KlO) KlStringMakeNoCopy(msgLen, pMsg);
	return ro;
    }
    else
	return NIL;
}

/* Return (but don't remove) first matching message form queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 *
 * Return: got message or NIL
 *
 * Wrapper: for int USPeekFirstMatchingMsg(int id, char *head, char **ppMsg, int *pLen);
 */
KlO
EmPeekFirstMatchingMsgKl(KlO id,
			 KlO head	/* String which must be the head of
			     matching message */ )
{
    KlO ro;
    char *pMsg;
    int msgLen;

    KlMustBeId(id, 0);
    KlMustBeStringOrConstString(head, 1);
    /* ?? TODO must return size too!!! */
    EmCheckError(USPeekFirstMatchingMsg(EmIdToCommId(KlNumToInt(id)), KlStringToCharPtr(head), &pMsg, &msgLen));
    if (msgLen) {
	ro = (KlO) KlStringMakeFromBytes(msgLen, pMsg);
	return ro;
    }
    else
	return NIL;
}


/* Return id associated to line nammed (lineName)
 * or NIL if no such line.
 *
 * Wrapper: for int USNameToId(char *lineName);
 */
KlO
EmNameToCommIdKl(KlO lineName)
{
    KlO ro;
    int id;

    KlMustBeStringOrConstString(lineName, 0);

    id = USNameToId(KlStringToCharPtr(lineName));
    if (id >= 0) {
	ro = (KlO) KlNumberMake(id);
	return ro;
    }
    return NIL;
}

/* Return name of line identified by id
 * or NIL if no such line.
 *
 * Wrapper for :char *USIdToName(int id);
 */
KlO
EmIdToNameKl(KlO id)
{
    KlO ro;
    char *s;

    KlMustBeId(id, 0);
    s = USIdToName(EmIdToCommId(KlNumToInt(id)));
    if (s) {
	ro = (KlO) KlStringMake(s);
	return ro;
    }
    else
	return NIL;
}

/* Init the communication module.
 */
void
EmCommInit(void)
{
    KlDeclareSubr(EmInitCommServKl, "init-comm", 2);
    KlDeclareSubr(EmCloseLineKl, "close-line", 1);
    KlDeclareSubr(EmAcceptNewLineKl, "accept-new-line", 0);
    KlDeclareSubr(EmWaitForNewLineKl, "wait-for-new-line", 0);
    KlDeclareSubr(EmSendKl, "send-msg", 2);
    KlDeclareSubr(EmFlushKl, "flush-line", 1);
    KlDeclareSubr(EmFlushAllClientsKl, "flush-all-lines", 0);
    KlDeclareSubr(EmPendingMsgKl, "pending-msg", 1);
    KlDeclareSubr(EmCheckPendingMsgKl, "check-pending-msg", 1);
    KlDeclareSubr(EmCheckAllLinesKl, "check-all-lines", 0);
    KlDeclareSubr(EmGetNextMsgKl, "get-next-msg", 1);
    KlDeclareSubr(EmPeekNextMsgKl, "peek-next-msg", 1);
    KlDeclareSubr(EmGetFirstMatchingMsgKl, "get-first-matching-msg", 2);
    KlDeclareSubr(EmPeekFirstMatchingMsgKl, "peek-first-matching-msg", 2);
    KlDeclareSubr(EmNameToCommIdKl, "name-to-commid", 1);
    KlDeclareSubr(EmIdToNameKl, "commid-to-name", 1);
}
