#include "emall.h"

static EmGBox *emGBoxGarbage = NULL;
static int emGBoxGarbageCount = 0;
static HashTable *NodGbHashTable;

typedef struct EmNodGBox {
    EmNode *pn;
    EmGroup *pg;
    EmGBox *pgb;
}         EmNodGBox;


/**************************************************************** Prototypes */

/* are two EmNodGBox struct equal ? */
static int EmGBOrdNodGb(EmNodGBox * png1, EmNodGBox * png2);

/* free a EmNodGBox struct */
static void EmFreeNodGb(EmNodGBox * png);

/* hash a EmNodGBox struct */
static int EmHashNodGb(EmNodGBox * png, int sz);

/* print a EmNodGBox struct */
static void EmPrintNodGb(EmNodGBox * png);

/* Print a EmGBox tree
 */
static void EmPrintGBTreeAux(EmGBox * pgb, int level);

/******************************************************************** Bodies */

/* are two EmNodGBox struct equal ? */
static int
EmGBOrdNodGb(EmNodGBox * png1, EmNodGBox * png2)
{
    return (png1->pn - png2->pn) + (png1->pg - png2->pg);
}

/* free a EmNodGBox struct */
static void
EmFreeNodGb(EmNodGBox * png)
{
    UFree(png);				/* ??TODO better allocations..
					 * garbage list? */
}


/* hash a EmNodGBox struct */
static int
EmHashNodGb(EmNodGBox * png, int sz)
{
    return ((int) png->pn + (int) png->pg) % sz;
}

/* print a EmNodGBox struct */
static void
EmPrintNodGb(EmNodGBox * png)
{
    EmPrintGroup(png->pg);
    EmPrintEmNode(png->pn);
    EmPrintGB(png->pgb);
}


/* Return a EmGBox state (human redably ;)
 */
char *
EmGBPrintState(EmGBox * pgb)
{
    if (EmGBRebuildP(pgb))
	printf("Rebuild ");
    if (EmGBSonRebuildP(pgb))
	printf("SonRebuild ");
    if (EmGBRefillP(pgb))
	printf("Refill ");
    if (EmGBRedrawP(pgb))
	printf("Redraw ");
    if (EmGBDiscardedP(pgb))
	printf("Discarded ");
    if (EmGBCachedP(pgb))
	printf("Cached ");
    printf("- ");
}


/* Return a EmGBox type  (human redably ;)
 */
char *
EmGBTypeToString(EmGBox * pgb)
{
    switch (pgb->type) {
	case GBRow:
	return "Row";
    case GBRowCut:
	return "RowCut";
    case GBColLeft:
	return "ColLeft";
    case GBColCenter:
	return "ColCenter";
    case GBIndex:
	return "Index";
    case GBUp:
	return "Up";
    case GBDown:
	return "Down";
    case GBVector:
	return "Vector";
    case GBMatrix:
	return "Matrix";
    case GBDiv:
	return "Div";
    case GBRoot:
	return "Root";
    case GBChar:
	return "Char";
    case GBSymb:
	return "Symb";
    case GBRootSymb:
	return "RootSymb";
    case GBHLine:
	return "HLine";
    case GBMultSymb:
	return "MultSymb";
    case GBUMinus:
	return "UMinus";
    case GBTemplate:
	return "Template";
    case GBContainer:
	return "Container";
    case GBBad:
    default:
	return "Bad";
    }
}

/* Check if father/son link is correct for this EmGBox and its father
 */
void
EmCheckLinkGB(EmGBox * pgb)
{
    EmGBox *pf, *pfs;

    pf = pgb->pFather;
    if (pf) {
	pfs = pf->pSon;
	while (pfs) {
	    if (pfs == pgb)
		return;			/* ok. pgb is really a child of pf */
	    pfs = pfs->pBrother;
	}
	UIError("EmCheckGBLink", "Misplaced child(hood ;). EmGBox tree corruption.");
    }
}

/* Print a short description of a given EmGBox
 */
void
EmPrintGB(EmGBox * pgb)
{
    if (pgb) {
	EmCheckLinkGB(pgb);
	printf("(GB %s ",
	       EmGBTypeToString(pgb));
	EmGBPrintState(pgb);
	printf("%s [", (pgb->pFather) ? "" : "*");
	EmPrintEmNode(pgb->pn);
	printf("])");
    }
    else {
	printf("(NULL)");
    }
}

/* Print a EmGBox tree
 */
static void
EmPrintGBTreeAux(EmGBox * pgb, int level)
{
    EmGBox *ps;
    int i;

    ps = pgb->pSon;
    for (i = 0; i < level; i++)
	putchar(' ');
    EmPrintGB(pgb);
    putchar('\n');
    while (ps) {
	EmPrintGBTreeAux(ps, level + 3);
	ps = ps->pBrother;
    }
}

/* Print a EmGBox tree
 */
void
EmPrintGBTree(EmGBox * pgb /* root of tree */ )
{
    putchar('\n');
    EmPrintGBTreeAux(pgb, 0);
}


/* Allocate a new EmGBox (not cleared).
 *
 * Allocation may be avoided if garbage list is not empty.
 */
EmGBox *
EmAllocEmGBox(void)
{
    EmGBox *pgb;

    if (emGBoxGarbageCount) {
	pgb = emGBoxGarbage;
	emGBoxGarbage = pgb->pSon;
	emGBoxGarbageCount--;
    }
    else
	pgb = UNew(EmGBox);
    bzero(pgb, sizeof(EmGBox));		/* ?? only during debug ? */
    EmGBRefillT(pgb);			/* well, this box will be modified of
					 * course */
    return pgb;
}

/* Free a EmGBox.
 *
 * Freed EmGBoxes are linked with their pSon pointers to form a garbage list.
 */
void
EmFreeEmGBox(EmGBox * pgb)
{
    UFree(pgb);
    return;				/* ??TODO debug */
    if (emGBoxGarbageCount < EM_MAX_EMGBOX_GARBAGE) {
	bzero(pgb, sizeof(EmGBox));	/* ?? only during debug ? */
	pgb->pSon = emGBoxGarbage;
	emGBoxGarbage = pgb;
	emGBoxGarbageCount++;
    }
    else {
	UFree(pgb);
    }
}



/* Find the EmGBox linked to a Node
   return the main EmGBox of this Node or NULL if none registred
 */
EmGBox *
EmNodeToEmGBox(EmGroup * pg		/* Group of Window to which EmGBox
	           are linked */ ,
	       EmNode * pn /* Node searched */ )
{
    EmNodGBox ngb, *pngb;

    assert(pg && pn);
    ngb.pg = pg;
    ngb.pn = pn;
    ngb.pgb = NULL;
    pngb = (EmNodGBox *) htSearchKey(NodGbHashTable, &ngb);
    if (pngb)
	return pngb->pgb;
    else
	return NULL;
}

/* Link a EmGBox to a Node (given the group of it's window)
 * Delete previous link between this node/window/EmGBox
 * free this EmGBox (and its sons sharing same node)
 */
EmGBox *
EmLinkEmGBoxToNode(EmGroup * pg		/* Group of Window to which EmGBox
		       will be linked */ ,
		   EmNode * pn /* Node to link */ ,
		   EmGBox * pgb /* EmGBox to link */ )
{

    EmGBox *poldgb;
    EmNodGBox ngb, *pngb;

    assert(pg && pn && pgb);
    poldgb = EmNodeToEmGBox(pg, pn);
    if (poldgb) {
	EmUnLinkEmGBoxToNode(pg, pn, poldgb);
    }

    pngb = UNew(EmNodGBox);
    pngb->pg = pg;
    pngb->pn = pn;
    pngb->pgb = pgb;
    htInsert(NodGbHashTable, pngb, True);
    return pgb;
}

/* Unlink a EmGBox to a Node (given the group of it's window)
 * Delete link between this node/window/EmGBox
 * free this EmGBox (and its sons sharing same node)
 */
void
EmUnLinkEmGBoxToNode(EmGroup * pg	/* Group of Window to which EmGBox
		         will be unlinked */ ,
		     EmNode * pn /* Node to unlink */ ,
		     EmGBox * rpgb /* EmGBox to unlink */ )
{
    EmGBox *pgb, *psgb, *pnsgb;
    EmNodGBox ngb, *pngb;

    assert(pg && pn && rpgb);

    /* delete all sons EmGBoxes sharing same EmNode */
    pgb = EmNodeToEmGBox(pg, pn);
    assert(rpgb == pgb);

    if (pgb) {
	psgb = pgb->pSon;
	while (psgb) {
	    pnsgb = psgb->pBrother;
	    psgb->pFather = NULL;
	    if (psgb->pn == pn) {
		/*
		 * that's a pseudo EmGBox for same EmNode as its father eg:
		 * symbol '+'
		 */
		EmFreeEmGBox(psgb);
	    }
	    psgb = pnsgb;
	}
	EmFreeEmGBox(pgb);
    }

    /* remove this entry from hashtable */
    ngb.pg = pg;
    ngb.pn = pn;
    ngb.pgb = NULL;
    htDelKey(NodGbHashTable, &ngb, True);
}



/* Find the set of EmGBox bounded by a rectangle
 */
EmGBox *
EmRectangleToEmGBoxes(EmGroup * pg /* Group to use for attributes */ ,
		      EmGBox * pgb /* Root box of filled tree */ ,
		      int vx		/* ul corner x of pgb (0 if pgb is
		          real root) */ ,
		      int vy		/* ul corner y of pgb (0 if pgb is
		          real root) */ ,
		      int ulx		/* upper left corner x (in virtual
		          window coordinates) */ ,
		      int uly		/* upper left corner y (in virtual
		          window coordinates) */ ,
		      int w /* width */ ,
		      int h /* height */ )
{
    EmGBox *psgb, *prgb;

    assert(pgb && pg);
    assert(w >= 0 && h >= 0);
    UNIW;				/* ?? TODO */
    return pgb;
}



/* This EmNode has been modified or added.
 * Set states of all boxes linked to this node and all their ancestors
 * to reflect this modification (this mean doing this for all runing windows)
 */
void
EmUpdateEmGBoxState(EmNode * pn)
{
    int i, nbw;
    EmGroup *pg;
    EmGBox *pgb;

    nbw = lNbElts(EmListOfWindows);
    /* for all windows */
    for (i = 0; i < nbw; i++) {
	pg = lLookNth(EmListOfWindows, i);
	if (pgb = EmNodeToEmGBox(pg, pn))
	    EmGBRebuildT(pgb);

	if (pn = pn->pf)		/* search for the EmGBox of father */
	    if (pgb = EmNodeToEmGBox(pg, pn))	/* father must be rebuild too */
		EmGBRebuildT(pgb);

	if (pn = pn->pf)		/* search for the EmGBox of
					 * greatfather */
	    pgb = EmNodeToEmGBox(pg, pn);
	while (pgb && !EmGBSonRebuildP(pgb)) {
	    /*
	     * ancestors must know that one of their descents has been
	     * modified
	     */
	    EmGBSonRebuildT(pgb);
	    pgb = pgb->pFather;
	}
    }
}


/* Wrapper:
 */
KlO
EmRectangleToEmGBoxesKl(int argc, KlO * argv)
{
    EmGBox *pgbres;
    int i;

    if (argc != 8)
	return KlBadNumberOfArguments((char *) argc);

    for (i = 0; i < argc; i++)
	KlMustBeIntOrConstInt(argv[i], i);

    pgbres = EmRectangleToEmGBoxes(KlNumToGrp(argv[0]), KlNumToEmGBox(argv[1]),
				   KlNumToInt(argv[2]), KlNumToInt(argv[3]),
				   KlNumToInt(argv[4]), KlNumToInt(argv[5]),
				   KlNumToInt(argv[6]), KlNumToInt(argv[7]));

    if (pgbres)
	return (KlO) KlNumberMake(pgbres);
    else
	return NIL;
}

/* Find the bigest EmGBox bounded by a rectangle
 */
EmGBox *
EmRectangleToBigestEmGBox(EmGroup * pg /* Group to use for attributes */ ,
			  EmGBox * pgb /* Root box of filled tree */ ,
			  int vx	/* ul corner x of pgb (0 if pgb is
			      real root) */ ,
			  int vy	/* ul corner y of pgb (0 if pgb is
			      real root) */ ,
			  int ulx	/* upper left corner x (in virtual
			      window coordinates) */ ,
			  int uly	/* upper left corner y (in virtual
			      window coordinates) */ ,
			  int w /* width */ ,
			  int h /* height */ )
{
    EmGBox *psgb, *prgb, *pbgb = NULL;
    int area = 0, tmp;

    assert(pgb && pg);
    assert(w >= 0 && h >= 0);
    if (GbBoundedP(pgb, vx, vy, ulx, uly, w, h)) {
	return pgb;
    }
    else {
	if ((psgb = pgb->pSon)) {
	    while (psgb) {
		if ((prgb = EmRectangleToBigestEmGBox(pg, psgb,
						      vx + psgb->x,
						      vy + psgb->y,
						      ulx, uly, w, h))) {
		    if ((tmp = (prgb->w * prgb->h)) > area) {
			area = tmp;
			pbgb = prgb;
		    }
		}
		psgb = psgb->pBrother;
	    }
	}
	else if (EmGBDiscardedP(pgb)) {
	    UNIW;
	}
	return pbgb;
    }
}

/* Wrapper:
 */
KlO
EmRectangleToBigestEmGBoxKl(int argc, KlO * argv)
{
    EmGBox *pgbres;
    int i;

    if (argc != 8)
	return KlBadNumberOfArguments((char *) argc);

    for (i = 0; i < argc; i++)
	KlMustBeIntOrConstInt(argv[i], i);

    pgbres = EmRectangleToBigestEmGBox(KlNumToGrp(argv[0]), KlNumToEmGBox(argv[1]),
				   KlNumToInt(argv[2]), KlNumToInt(argv[3]),
				   KlNumToInt(argv[4]), KlNumToInt(argv[5]),
				  KlNumToInt(argv[6]), KlNumToInt(argv[7]));
    if (pgbres)
	return (KlO) KlNumberMake(pgbres);
    else
	return NIL;
}


/* Find the highest EmGBox bounded by a rectangle
 * highest means that the EmNode the EmGBox is orginated from is higher in tree
 * than other bounded EmGBoxes
 */
EmGBox *
EmRectangleToHighestEmGBox(EmGroup * pg /* Group to use for attributes */ ,
			   EmGBox * pgb /* Root box of filled tree */ ,
			   int vx	/* ul corner x of pgb (0 if pgb is
			       real root) */ ,
			   int vy	/* ul corner y of pgb (0 if pgb is
			       real root) */ ,
			   int ulx	/* upper left corner x (in virtual
			       window coordinates) */ ,
			   int uly	/* upper left corner y (in virtual
			       window coordinates) */ ,
			   int w /* width */ ,
			   int h /* height */ )
{
    EmGBox *psgb, *prgb, *pbgb = NULL;
    int level = INT_MAX;

    assert(pgb && pg);
    assert(w >= 0 && h >= 0);
    if (GbBoundedP(pgb, vx, vy, ulx, uly, w, h)) {
	return pgb;
    }
    else {
	if ((psgb = pgb->pSon)) {
	    while (psgb) {
		if ((prgb = EmRectangleToHighestEmGBox(pg, psgb,
						       vx + psgb->x,
						       vy + psgb->y,
						       ulx, uly, w, h))) {
		    if (prgb->pn->level < level) {
			level = prgb->pn->level;
			pbgb = prgb;
		    }
		}
		psgb = psgb->pBrother;
	    }
	}
	else if (EmGBDiscardedP(pgb)) {
	    UNIW;
	}
	return pbgb;
    }
}

/* Wrapper:
 */
KlO
EmRectangleToHighestEmGBoxKl(int argc, KlO * argv)
{
    EmGBox *pgbres;
    int i;

    if (argc != 8)
	return KlBadNumberOfArguments((char *) argc);

    for (i = 0; i < argc; i++)
	KlMustBeIntOrConstInt(argv[i], i);

    pgbres = EmRectangleToHighestEmGBox(KlNumToGrp(argv[0]), KlNumToEmGBox(argv[1]),
				   KlNumToInt(argv[2]), KlNumToInt(argv[3]),
				   KlNumToInt(argv[4]), KlNumToInt(argv[5]),
				  KlNumToInt(argv[6]), KlNumToInt(argv[7]));
    if (pgbres)
	return (KlO) KlNumberMake(pgbres);
    else
	return NIL;
}



/* Find the EmGBox bounding a point
 */
EmGBox *
EmCoordToEmGBox(EmGroup * pg /* Group to use for attributes */ ,
		EmGBox * pgb /* Root box of filled tree */ ,
		int vx			/* ul corner x of pgb (0 if pgb is
		    real root) */ ,
		int vy			/* ul corner y of pgb (0 if pgb is
		    real root) */ ,
		int x			/* point x given in virtual window
		    coordinates */ ,
		int y			/* point ygiven in virtual window
		    coordinates */ )
{
    EmGBox *psgb, *prgb;

    assert(pgb && pg);
    if (GbBoundP(pgb, vx, vy, x, y)) {
	if ((psgb = pgb->pSon)) {
	    while (psgb) {
		if ((prgb = EmCoordToEmGBox(pg, psgb,
					    vx + psgb->x, vy + psgb->y,
					    x, y))) {
		    return prgb;
		}
		psgb = psgb->pBrother;
	    }
	}
	else if (EmGBDiscardedP(pgb)) {
	    UNIW;
	}
	return pgb;
    }
    return (EmGBox *) NULL;
}

/* Wrapper:
 */
KlO
EmCoordToEmGBoxKl(int argc, KlO * argv)
{
    EmGBox *pgbres;
    int i;

    if (argc != 6)
	return KlBadNumberOfArguments((char *) argc);

    for (i = 0; i < argc; i++)
	KlMustBeIntOrConstInt(argv[i], i);

    pgbres = EmCoordToEmGBox(KlNumToGrp(argv[0]), KlNumToEmGBox(argv[1]),
			     KlNumToInt(argv[2]), KlNumToInt(argv[3]),
			     KlNumToInt(argv[4]), KlNumToInt(argv[5]));
    if (pgbres)
	return (KlO) KlNumberMake(pgbres);
    else
	return NIL;
}

/* Return the node associated to a EmGBox
 */
EmNode *
EmGBoxToNode(EmGroup * pg /* Group to use for attributes */ ,
	     EmGBox * pgb /* Searched EmGBox */ )
{
    EmGBox *psgb, *prgb;

    assert(pgb && pg);
    return pgb->pn;
}


/* Wrapper:
 */
KlO
EmGBoxToNodeKl(KlO pg /* Group to use for attributes */ ,
	       KlO pgb /* Searched EmGBox */ )
{
    KlMustBeIntOrConstInt(pg, 0);
    KlMustBeIntOrConstInt(pgb, 1);
    return (KlO) KlNumberMake(EmGBoxToNode(KlNumToGrp(pg), KlNumToEmGBox(pgb)));
}



/* Must fill all default attributes concerning EmGBoxes.
 */
void
EmGBoxFDA(EmGroup * pg /* Group to fill */ ,
	  EmNode * pn /* Node to fill */ )
{
    EmGBoxFDABuild(pg, pn);
    EmGBoxFDAFill(pg, pn);
    EmGBoxFDADraw(pg, pn);
}

/* init the EmGBox module
 */
void
EmGBoxInit(void)
{
    /* Init the garbage queue */
    emGBoxGarbageCount = 0;
    emGBoxGarbage = EmAllocEmGBox();

    NodGbHashTable = htCreate(1000, keySame, EmGBOrdNodGb, EmFreeNodGb, EmHashNodGb, EmPrintNodGb);

    KlDeclareSubr(EmCoordToEmGBoxKl, "coord-to-gbox", NARY);
    KlDeclareSubr(EmGBoxToNodeKl, "gbox-to-node", 2);
    KlDeclareSubr(EmRectangleToBigestEmGBoxKl, "rectangle-to-gboxes", NARY);
    KlDeclareSubr(EmRectangleToBigestEmGBoxKl, "rectangle-to-bigest-gbox", NARY);
    KlDeclareSubr(EmRectangleToHighestEmGBoxKl, "rectangle-to-highest-gbox", NARY);


}
