#include <memory.h>
#include "emall.h"


/**************************************************************** Prototypes */

/* Alloc a node
 */
static EmNode * EmAllocNode ( void );
/* Free a node
 */
static void EmFreeNode ( EmNode * pn );
/* Alloc an arry of nodes
 * ?? TODO optimize using a pool of already allocated arrays
 * should be much better than reallocating */
static EmNode ** EmAllocNodeArray ( int nbsn );
/* Realloc an array of nodes
 */
static EmNode ** EmReallocNodeArray ( EmNode ** aps, int oldNbsn, int nbsn );
/* Free an array of nodes
 */
static void EmFreeNodeArray ( EmNode ** aps, int nbsn );
/* Convert a key (char) to a type
 */
static EmNodeType EmKeyToNodeType ( char *key );
/* auxiliary function
 */
static void EmDelTreeAux ( EmNode * pn );
/* Define some constants used from Klone... type of nodes
 */
static void EmDefTreeConstant ( void );

/******************************************************************** Bodies */

/* Alloc a node
 */
static EmNode *
EmAllocNode(void)
{
    return (EmNode *) UMallocMsg(sizeof(EmNode), "EmAllocNode");
}

/* Free a node
 */
static void
EmFreeNode(EmNode * pn)
{
    if (pn->nbsn) {
	if (pn->d.aps)
	    EmFreeNodeArray(pn->d.aps, pn->nbsn);
    }
    else {
	if (pn->d.value)
	    KlDecRef(pn->d.value);
    }
    free(pn);
}

/* Alloc an arry of nodes
 * ?? TODO optimize using a pool of already allocated arrays
 * should be much better than reallocating */
static EmNode **
EmAllocNodeArray(int nbsn)
{
    return (EmNode **) UMallocMsg(sizeof(EmNode *) * nbsn, "EmAllocNodeArray");
}

/* Realloc an array of nodes
 */
static EmNode **
EmReallocNodeArray(EmNode ** aps, int oldNbsn, int nbsn)
{
    EmNode **naps;

    if (nbsn >= oldNbsn) {
	return (EmNode **) URealloc(aps, nbsn * sizeof(EmNode *));
    }
    else {
	naps = EmAllocNodeArray(nbsn);
	memcpy(naps, aps, nbsn * sizeof(EmNode *));
	EmFreeNodeArray(aps, oldNbsn);
	return naps;
    }
}

/* Free an array of nodes
 */
static void
EmFreeNodeArray(EmNode ** aps, int nbsn)
{
    free(aps);
}

/* Wrapper: give a node and return a human readable type
 */
KlO
EmNodeTypeToStringKl(KlO idn)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    ro = (KlO) KlStringMake(EmNodeTypeToString(EmTypeOfNode(EmIdToNodePtr(idn))));
    return ro;
}


/* Convert a node type to its human readable form :)
 */
char *
EmNodeTypeToString(EmNodeType type)
{
    switch (type) {
	case NTBad:
	return "Bad";
    case NTPlus:
	return "Plus";
    case NTMult:
	return "Mult";
    case NTDiv:
	return "Div";
    case NTUMinus:
	return "UMinus";
    case NTEq:
        return "Eq";
    case NTRoot:
	return "Root";
    case NTPow:
	return "Pow";
    case NTSub:
	return "Sub";
    case NTIntegral:
	return "Integral";
    case NTFunc:
	return "Func";
    case NTColumn:
	return "Column";
    case NTVector:
	return "Vector";
    case NTMatrix:
	return "Matrix";
    case NTString:
	return "String";
    case NTNumber:
	return "Number";
    case NTLetter:
	return "Letter";
    case NTDigit:
	return "Digit";
    case NTSymbol:
	return "Symbol";
    case NTParenth:
	return "Parenth";
    case NTTemplate:
	return "Template";
    case NTContainer:
	return "Container";

    default:
	return "Unknown";
    }
}

/* Wrapper: give a char and return its readable type
 */
KlO
EmKeyToReadableNodeTypeKl(KlO key)
{
    KlO ro;

    KlMustBeStringOrConstString(key, 0);
    ro = (KlO) KlStringMake(EmNodeTypeToString(EmKeyToNodeType(KlStringToCharPtr(key))));
    return ro;
}

/* Wrapper: give a char and return its type
 */
KlO
EmKeyToNodeTypeKl(KlO key)
{
    KlO ro;

    KlMustBeStringOrConstString(key, 0);
    ro = (KlO) KlNumberMake(EmKeyToNodeType(KlStringToCharPtr(key)));
    return ro;
}

/* Convert a key (char) to a type
 */
static EmNodeType
EmKeyToNodeType(char *key)
{
    switch (*key) {
	case '+':
	return NTPlus;
    case '*':
	return NTMult;
    case '/':
	return NTDiv;
    case '^':
	return NTPow;
    case '-':
	return NTUMinus;
    case '_':
	return NTSub;
    case '(':
        return NTParenth;
    case '=':
        return NTEq;
    default:
	return NTBad;
    }
}

/* Print a node 
 */
void
EmPrintEmNode(EmNode * pn)
{
   if (pn) {
	printf("(%s ", EmNodeTypeToString(pn->type));
	if (pn->nbsn == 0)
	    if (pn->d.value) {
		KlPrint(pn->d.value);
	    }
	    else
		printf("NULL");
	printf(")");
    }
    else
	printf("Null Node");
}


/* Print a node tree
 */
void
EmPrintTree(EmNode * pn)
{
    int i;

    if (pn) {
	printf("(%s ", EmNodeTypeToString(pn->type));
	for (i = 0; i < pn->nbsn; i++) {
	    EmPrintTree(pn->d.aps[i]);
	}
	if (pn->nbsn == 0)
	    if (pn->d.value) {
		KlPrint(pn->d.value);
	    }
	    else
		printf("NULL");
	printf(")");
    }
    else
	printf("Null Node");
}

/* Wrapper:
 */
KlO
EmPrintTreeKl(KlO id)
{
    KlMustBeIntOrConstInt(id, 0);
    EmPrintTree(EmIdToNodePtr(id));
    return id;
}

/* Create a new node.
 *
 * Return: Adress of this node which is used as Id for this node
 */
EmNode *
EmCreateNode(EmNodeType type /* Type of node the created node. */ )
{
    EmNode *pn;

    pn = EmAllocNode();
    pn->type = type;
    pn->nbsn = 0;
    pn->pf = NULL;
    pn->level = 0;			/* default is root of a new tree */
    pn->d.aps = NULL;
    pn->d.value = NULL;
    return pn;
}

/* Wrapper:
 */
KlO
EmCreateNodeKl(KlO type)
{
    KlO ro;

    KlMustBeIntOrConstInt(type, 0);
    ro = (KlO) KlNumberMake(EmNodePtrToId(EmCreateNode((EmNodeType) ((KlNumber) type)->number)));
    return ro;
}

/* auxiliary function
 */
static void
EmDelTreeAux(EmNode * pn)
{
    int i;

    for (i = 0; i < pn->nbsn; i++) {
	EmDelTreeAux(pn->d.aps[i]);
    }
    EmFreeNode(pn);
}

/* Delete a tree (pn)
 *
 * Delete the node pn and all its descendants.
 * This node is discarded from its father array of sons (which is not compacted).
 * Warning: The attributes pined to this tree must be explicitly freed.
 */
void
EmDelTree(EmNode * pn)
{
    EmRemoveSonNode(pn->pf, pn);
    EmDelTreeAux(pn);
}

/* Wrapper:
 */
KlO
EmDelTreeKl(KlO id)
{
    KlMustBeIntOrConstInt(id, 0);
    EmDelTree(EmIdToNodePtr(id));
    return id;
}

/* Delete a node.
 *
 * The node is discarded from its father array of sons (which is not compacted)
 * If deleted node has sons, these poor orphans must have been relinked
 * somewhere (eg with EmAddSonNode* )
 *
 * Warning: The attributes pined to this node must be explicitly freed
 */
void
EmDelNode(EmNode * pn /* The node to delete */ )
{
    EmRemoveSonNode(pn->pf, pn);
    EmFreeNode(pn);
}

/* Wrapper:
 */
KlO
EmDelNodeKl(KlO id)
{
    KlMustBeIntOrConstInt(id, 0);
    EmDelNode(EmIdToNodePtr(id));
    return id;
}

/* Add a Son (ps) to a Node (pn)
 *
 * Return: Rank of son
 */
int
EmAddSonNode(EmNode * pn,
	     EmNode * ps /* is added as last son of pn */ )
{
    EmAddSonNodeNth(pn, ps, pn->nbsn);
    return pn->nbsn - 1;		/* warning... do not change
					 * reallocation of nbsn (must be
					 * incremented by one ) */
}

/* Wrapper:
 */
KlO
EmAddSonNodeKl(KlO idn, KlO ids)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(ids, 1);
    ro = (KlO) KlNumberMake(EmAddSonNode(EmIdToNodePtr(idn), EmIdToNodePtr(ids)));
    return ro;
}

/* Add the Nth (num) Son (ps) to a Node (pn).
 *
 * Current sons are right shifted if needed
 */
void
EmAddSonNodeNth(EmNode * pn, EmNode * ps, int num)
{
    EmNode **aps;

    assert(num >= 0);
    if ((!pn->nbsn) && pn->d.value) {
	KlDecRef(pn->d.value);
	pn->d.aps = NULL;
    }
    if (num < pn->nbsn) {		/* not at the end */
	if (pn->d.aps[num]) {		/* realloc+1 and shift right */
	    aps = EmAllocNodeArray(pn->nbsn + 1);
	    memcpy(aps, pn->d.aps, num * sizeof(EmNode *));
	    memcpy(aps + num + 1, pn->d.aps + num, (pn->nbsn - num) * sizeof(EmNode *));
	    EmFreeNodeArray(pn->d.aps, pn->nbsn);
	    pn->d.aps = aps;
	    pn->nbsn++;
	}
	/* else no realloc no shift */
    }
    else {				/* realloc (no shift) */
	pn->d.aps = EmReallocNodeArray(pn->d.aps, pn->nbsn, num + 1);
	pn->nbsn = num + 1;
    }
    pn->d.aps[num] = ps;		/* link son in father */
    ps->pf = pn;			/* link father in son */
    if ((ps->level) != (pn->level + 1)) {	/* must modify levels in
						 * whole subtree of ps */
	EmChangeTreeLevel(ps, pn->level + 1);
    }
    EmGbUpdateState(ps); /* reset state of ps and ps's ancestor to reflect a need rebuild */
}

/* Wrapper:
 */
KlO
EmAddSonNodeNthKl(KlO idn, KlO ids, KlO num)
{
    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(ids, 1);
    KlMustBeIntOrConstInt(num, 2);
    EmAddSonNodeNth(EmIdToNodePtr(idn), EmIdToNodePtr(ids), KlNumToInt(num));
    return idn;
}

/* Replace the Nth (num) Son (ps) to a Node (pn)
 *
 * Current son number (num) and all is descendants are freed with EmDelTree()
 */
void
EmReplaceSonNodeNth(EmNode * pn, EmNode * ps, int num)
{
    assert(num >= 0);
    if ((pn->nbsn > num) && (pn->d.aps[num])) {
	EmDelTree(pn->d.aps[num]);
	pn->d.aps[num] = NULL;
    }
    EmAddSonNodeNth(pn, ps, num);
}

/* Wrapper:
 */
KlO
EmReplaceSonNodeNthKl(KlO idn, KlO ids, KlO num)
{
    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(ids, 1);
    KlMustBeIntOrConstInt(num, 2);
    EmReplaceSonNodeNth(EmIdToNodePtr(idn), EmIdToNodePtr(ids), KlNumToInt(num));
    return idn;
}

/* Add an array (aps) of (num) sons to a node (pn)
 *
 * Warning: Array aps must be freed by caller
 *
 * Return: number of sons
 */
int
EmAddSonNodes(EmNode * pn, EmNode ** aps, int num)
{
    UIError("EmAddSonNodes", "not implemented");
    return 0;
}

/* Wrapper:
 */
KlO
EmAddSonNodesKl(KlO idn, KlO aps, KlO num)
{
    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(aps, 1);
    KlNumberMake(EmAddSonNodes(EmIdToNodePtr(idn), (EmNode **) EmIdToNodePtr(aps), KlNumToInt(num)));
    return idn;
}

/* Remove a son (ps) from node (pn)
 *
 * Nothing is freed (tree (ps) may be linked to another father).
 * However ps may be freed explicitly (eg: with EmDelNode or EmDelTree)
 */
void
EmRemoveSonNode(EmNode * pn, EmNode * ps)
{
    int i;

    for (i = 0; i < pn->nbsn; i++) {
	if (pn->d.aps[i] == ps) {
	    if (i == pn->nbsn - 1) {
		pn->d.aps = EmReallocNodeArray(pn->d.aps, pn->nbsn, i);
		pn->nbsn--;
	    }
	    else {
		pn->d.aps[i] = NULL;
	    }
	    if (!pn->nbsn)
		pn->d.aps = NULL;
	    break;
	}
    }
}

/* Wrapper:
 */
KlO
EmRemoveSonNodeKl(KlO idn, KlO ids)
{
    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(ids, 1);
    EmRemoveSonNode(EmIdToNodePtr(idn), EmIdToNodePtr(ids));
    return idn;
}

/* Remove the Nth (num) son (ps) from node (pn).
 */
void
EmRemoveSonNodeNth(EmNode * pn, int num)
{
    assert(num >= 0);
    assert(num < pn->nbsn);
    if (num == pn->nbsn - 1) {
	pn->d.aps = EmReallocNodeArray(pn->d.aps, pn->nbsn, num);
	pn->nbsn = num;
    }
    else {
	pn->d.aps[num] = NULL;
    }
}

/* Wrapper:
 */
KlO
EmRemoveSonNodeNthKl(KlO idn, KlO num)
{
    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(num, 1);
    EmRemoveSonNodeNth(EmIdToNodePtr(idn), KlNumToInt(num));
    return idn;
}

/* Given a node compute its rank as a son
 * return -1 if the son is not found as a child of this father
 */
int
EmFindRankOfSon(EmNode * ps)
{
    int i;

    assert(ps);
    if (!ps->pf)
	return 0;
    for (i = 0; i < ps->pf->nbsn; i++)
	if (ps->pf->d.aps[i] == ps)
	    return i;
    return -1;
}

/* Wrapper:
 */
KlO
EmFindRankOfSonKl(KlO ids)
{
    KlO ro;

    KlMustBeIntOrConstInt(ids, 0);
    ro = (KlO) KlNumberMake(EmFindRankOfSon(EmIdToNodePtr(ids)));
    return ro;
}

/* Insert a node (pi) between a father (pf) and its son (ps)
 * thus adding 1 to the tree level of the son
 */
EmNode *
EmInsertNode(EmNode * pf, EmNode * ps, EmNode * pi)
{
    int nos;

    assert(pf && ps && pi);
    nos = EmFindRankOfSon(ps);
    assert(nos >= 0);
    pf->d.aps[nos] = pi;
    pi->pf = pf;

    EmAddSonNode(pi, ps);
    return ps;
}

/* Wrapper:
 */
KlO
EmInsertNodeKl(KlO idf, KlO ids, KlO idi)
{
    KlMustBeIntOrConstInt(idf, 0);
    KlMustBeIntOrConstInt(ids, 1);
    KlMustBeIntOrConstInt(idi, 2);
    EmInsertNode(EmIdToNodePtr(idf), EmIdToNodePtr(ids), EmIdToNodePtr(idi));
    return idi;
}

/* Insert a node (pi) between a father (pf) and its son (ps)
 * thus adding 1 to the tree level of the son.
 *
 * Warning: Not tested.
 */
EmNode *
EmInsertNodeNth(EmNode * pf, EmNode * ps, EmNode * pi, int num)
{
    int nos;

    assert(pf && ps && pi);
    nos = EmFindRankOfSon(ps);
    assert(nos >= 0);
    pf->d.aps[nos] = pi;
    pi->pf = pf;

    EmAddSonNodeNth(pi, ps, num);
    return ps;
}

/* Wrapper:
 */
KlO
EmInsertNodeNthKl(KlO idf, KlO ids, KlO idi, KlO num)
{
    KlMustBeIntOrConstInt(idf, 0);
    KlMustBeIntOrConstInt(ids, 1);
    KlMustBeIntOrConstInt(idi, 2);
    KlMustBeIntOrConstInt(num, 3);
    EmInsertNodeNth(EmIdToNodePtr(idf), EmIdToNodePtr(ids), EmIdToNodePtr(idi), KlNumToInt(num));
    return idi;
}

/* Change the level of all nodes in a tree (pn).
 *
 * NewLevel becomes the root level and all subtree nodes are updated acordingly
 */
void
EmChangeTreeLevel(EmNode * pn, int newLevel)
{
    int i;

    pn->level = newLevel;
    for (i = 0; i < pn->nbsn; i++) {
	if (pn->d.aps[i]) {
	    EmChangeTreeLevel(pn->d.aps[i], newLevel + 1);
	}
    }
}

/* Wrapper:
 */
KlO
EmChangeTreeLevelKl(KlO idn, KlO newLevel)
{
    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(newLevel, 1);
    EmChangeTreeLevel(EmIdToNodePtr(idn), KlNumToInt(newLevel));
    return idn;
}

/* Compact the sons of a node
 *
 * Remove empty sons from array of sons
 *
 * Return: the number of non empty sons
 */
int
EmCompactSonNodes(EmNode * pn)
{
    int lne = 0, i;

    for (i = 0; i < pn->nbsn; i++) {
	if (pn->d.aps[i]) {
	    pn->d.aps[lne] = pn->d.aps[i];
	    lne++;
	}
    }
    if (pn->nbsn != lne) {
	pn->d.aps = EmReallocNodeArray(pn->d.aps, pn->nbsn, lne);
	pn->nbsn = lne;
    }
    return lne;
}

/* Wrapper:
 */
KlO
EmCompactSonNodesKl(KlO idn)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    ro = (KlO) KlNumberMake(EmCompactSonNodes(EmIdToNodePtr(idn)));
    return ro;
}

/* Return: the type of a node.
 */
EmNodeType
EmTypeOfNode(EmNode * pn /* The node */ )
{
    return pn->type;
}

/* Wrapper:
 */
KlO
EmTypeOfNodeKl(KlO idn)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    ro = (KlO) KlNumberMake(EmTypeOfNode(EmIdToNodePtr(idn)));
    return ro;
}

/* Set type of a node
 */
EmNode *
EmSetNodeType(EmNode * pn /* The node */ ,
	      EmNodeType type /* The new type */ )
{
    pn->type = type;

    return pn;
}

/* Wrapper:
 */
KlO
EmSetNodeTypeKl(KlO idn, KlO type)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(type, 0);
    ro = (KlO) KlNumberMake(EmNodePtrToId(EmSetNodeType(EmIdToNodePtr(idn), (EmNodeType) (((KlNumber) type)->number))));
    return ro;
}

/* Return: the father of a node.
 */
EmNode *
EmFatherNode(EmNode * pn /* The node */ )
{
    if (pn)
	return pn->pf;
    else
	return (EmNode *) NULL;
}

/* Wrapper:
 */
KlO
EmFatherNodeKl(KlO idn)
{
    KlO ro;
    EmId pf;

    KlMustBeIntOrConstInt(idn, 0);
    pf = EmNodePtrToId(EmFatherNode(EmIdToNodePtr(idn)));
    if (pf) {
	ro = (KlO) KlNumberMake(pf);
	return ro;
    }
    return NIL;

}


/* Return: Value stored in a node.
 */
KlO
EmGetNodeValue(EmNode * pn /* The node */ )
{
    if (pn->nbsn) {			/* that's not a leaf */
	return NIL;
    }
    else {
	return pn->d.value;
    }
}

/* Wrapper:
 */
KlO
EmGetNodeValueKl(KlO idn)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    ro = EmGetNodeValue(EmIdToNodePtr(idn));
    if (ro) {
	return ro;
    }
    return NIL;
}

/* Store a value in a node
 */
KlO
EmSetNodeValue(EmNode * pn /* The node */ ,
	       KlO val /* The value to store */ )
{
    assert(!pn->nbsn);			/* avoid an aps trash (assert it's a
					 * leaf) */
    if (pn->d.value)
	KlDecRef(pn->d.value);
    pn->d.value = val;
    KlIncRef(val);
    return pn->d.value;
}

/* Wrapper:
 */
KlO
EmSetNodeValueKl(KlO idn, KlO val)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    ro = EmSetNodeValue(EmIdToNodePtr(idn), val);
    return ro;
}


/* Return: the number of sons of a node.
 */
int
EmNbSonNodes(EmNode * pn /* The node */ )
{
    int i, nbs = 0;

    for (i = 0; i < pn->nbsn; i++)
	if (pn->d.aps[i])
	    nbs++;
    return nbs;
}

/* Wrapper:
 */
KlO
EmNbSonNodesKl(KlO idn)
{
    KlO ro;

    KlMustBeIntOrConstInt(idn, 0);
    ro = (KlO) KlNumberMake(EmNbSonNodes(EmIdToNodePtr(idn)));
    return ro;
}

/* Return: the Nth (n) son of a node (pn)
 */
EmNode *
EmNthSonNode(EmNode * pn /* The node */ , int n /* Rank of node to return */ )
{
    assert(n >= 0);
    assert(n < pn->nbsn);
    return pn->d.aps[n];
}

/* Wrapper:
 */
KlO
EmNthSonNodeKl(KlO idn, KlO num)
{
    KlO ro;
    EmId ps;

    KlMustBeIntOrConstInt(idn, 0);
    KlMustBeIntOrConstInt(num, 1);
    ps = EmNodePtrToId(EmNthSonNode(EmIdToNodePtr(idn), KlNumToInt(num)));
    if (ps) {
	ro = (KlO) KlNumberMake(ps);
	return ro;
    }
    return NIL;
}

/* Define some constants used from Klone... type of nodes
 */
static void
EmDefTreeConstant(void)
{
    KlConstantMake("tree:Bad", KlNumberMake(NTBad));
    KlConstantMake("tree:Plus", KlNumberMake(NTPlus));
    KlConstantMake("tree:Mult", KlNumberMake(NTMult));
    KlConstantMake("tree:Div", KlNumberMake(NTDiv));
    KlConstantMake("tree:UMinus", KlNumberMake(NTUMinus));
    KlConstantMake("tree:Eq", KlNumberMake(NTEq));
    KlConstantMake("tree:Exp", KlNumberMake(NTExp));
    KlConstantMake("tree:Sub", KlNumberMake(NTSub));
    KlConstantMake("tree:Root", KlNumberMake(NTRoot));
    KlConstantMake("tree:Pow", KlNumberMake(NTPow));
    KlConstantMake("tree:Integral", KlNumberMake(NTIntegral));
    KlConstantMake("tree:Func", KlNumberMake(NTFunc));
    KlConstantMake("tree:Column", KlNumberMake(NTColumn));
    KlConstantMake("tree:Vector", KlNumberMake(NTVector));
    KlConstantMake("tree:Matrix", KlNumberMake(NTMatrix));
    KlConstantMake("tree:String", KlNumberMake(NTString));
    KlConstantMake("tree:Number", KlNumberMake(NTNumber));
    KlConstantMake("tree:Letter", KlNumberMake(NTLetter));
    KlConstantMake("tree:Digit", KlNumberMake(NTDigit));
    KlConstantMake("tree:Symbol", KlNumberMake(NTSymbol));
    KlConstantMake("tree:Parenth", KlNumberMake(NTParenth));
    KlConstantMake("tree:Template", KlNumberMake(NTTemplate));
    KlConstantMake("tree:Container", KlNumberMake(NTContainer));
}


/* Initialize all tree manipulating functions for Klone
 */
void
EmTreeInit(void)
{
    EmDefTreeConstant();
    KlDeclareSubr(EmCreateNodeKl, "create-node", 1);
    KlDeclareSubr(EmDelTreeKl, "del-tree", 1);
    KlDeclareSubr(EmDelNodeKl, "del-node", 1);
    KlDeclareSubr(EmAddSonNodeKl, "add-son", 2);
    KlDeclareSubr(EmAddSonNodeNthKl, "add-son-nth", 3);
    KlDeclareSubr(EmReplaceSonNodeNthKl, "replace-son-nth", 3);
    KlDeclareSubr(EmAddSonNodesKl, "add-sons", 3);
    KlDeclareSubr(EmRemoveSonNodeKl, "remove-son", 2);
    KlDeclareSubr(EmInsertNodeKl, "insert-node", 3);
    KlDeclareSubr(EmInsertNodeNthKl, "insert-node-nth", 4);
    KlDeclareSubr(EmFindRankOfSonKl, "rank-of-son", 1);
    KlDeclareSubr(EmRemoveSonNodeNthKl, "remove-son-nth", 2);
    KlDeclareSubr(EmChangeTreeLevelKl, "change-tree-level", 2);
    KlDeclareSubr(EmCompactSonNodesKl, "compact-sons", 1);
    KlDeclareSubr(EmTypeOfNodeKl, "node-type", 1);
    KlDeclareSubr(EmKeyToNodeTypeKl, "key-to-node-type", 1);
    KlDeclareSubr(EmKeyToReadableNodeTypeKl, "key-to-readable-node-type", 1);
    KlDeclareSubr(EmNodeTypeToStringKl, "node-type-to-string", 1);
    KlDeclareSubr(EmSetNodeValueKl, "set-node-value", 2);
    KlDeclareSubr(EmGetNodeValueKl, "get-node-value", 1);
    KlDeclareSubr(EmFatherNodeKl, "father-node", 1);
    KlDeclareSubr(EmNbSonNodesKl, "nb-sons", 1);
    KlDeclareSubr(EmNthSonNodeKl, "nth-son", 2);
    KlDeclareSubr(EmSetNodeTypeKl, "set-node-type", 2);
}
