#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include "emall.h"


/* used as a tick marker */
int EmTpTick = 0;

/* class for TD struct */
KlStructClass EmTdClass;

/* a array of prioritized queues of threads */
static List *EmTpRunQA[EM_TP_MAX_PRI_QUEUE + 1];

/* used to store the main thread (ie scheduler) struct when calling a real thread */
static Td EmTpMainThread;

/* the global list of thread. all created threads are in this list */
static List *EmTpThreadList = NULL;


/**************************************************************** Prototypes */

/* Free a thread data structure
 */
static void 
EmTpTdFree(Td * ptd);

/* Set the tick used by Yield to see if enogh time is ellapsed since
 * last give up to scheduler
 */
static void 
EmTpSetTick(void);

/* Init the tick alarm
 */
static void 
EmTpInitTimer(void);

/* Call this before any other thread function
 */
static void 
EmTpInitInternal(void);

/* static function not documented...
 */
static void 
EmTpOnly(void *pu, void *pt, qt_userf_t * f);

/* static function not documented...
 */
static void 
EmTpStartHelp(qt_t * old, void *ignore0, void *ignore1);

/* static function not documented...
 */
static void 
EmTpYieldHelp(qt_t * sp, void *old, void *null);

/* static function not documented...
 */
static void 
EmTpAbortHelp(qt_t * sp, void *old, void *null);

/* Make klone class for thread data (Td)
 */
static void 
EmTdClassMake(void);

/* Define some constants used from Klone... priorities
 */
static void 
EmDefTpConstant(void);

/******************************************************************** Bodies */




/* Free a thread data structure
 */
static void
EmTpTdFree(Td * ptd)
{
    if (ptd->sto) {
	free(ptd->sto);
	free(ptd);
    }
}

/* Print thread data
 */
void
EmTpTdPrint(Td * ptd)
{
    printf("Td(pri=%d rest/need=%d/%d %s)",
	   ptd->priority,
	   ptd->restartRefresh,
	   ptd->needRefresh,
	   (ptd->status == EM_TP_DYING) ? "DYING" : (ptd->status == EM_TP_RUNING) ? "RUNING" : "UNKNOWN"
    );
}

/* ?? just for debuging
 */
void
EmTpRunQAPrint(void)
{
    int priority;

    for (priority = EM_TP_MAX_PRI_QUEUE; priority >= 0; priority--) {
	lPrint(EmTpRunQA[priority]);
    }
}

/* This window will be redrawn as soon as possible (during next scheduling)
 */
KlO
EmTpRestartRefreshKl(KlO winId)
{
    Td *ptd;

    KlMustBeIntOrConstInt(winId, 0);
    ptd = EmWinToThread(KlNumToLong(winId));
    EmTpRestartRefresh(ptd, True);
    return NIL;
}

/* This window is beeing drawn... so call its refreshing thread during
 * next scheduling
 */
KlO
EmTpNeedRefreshKl(KlO winId)
{
    Td *ptd;

    KlMustBeIntOrConstInt(winId, 0);
    ptd = EmWinToThread(KlNumToLong(winId));
    EmTpNeedRefresh(ptd, True);
    return NIL;
}

/* Change the priority of a thread
 */
void
EmTpChangePriority(Td * ptd, int newPri)
{
    lDel(EmTpRunQA[ptd->priority], ptd, False);
    ptd->priority = newPri;
    lAdd(EmTpRunQA[newPri], ptd);
}

/* Wrapper: for EmTpChangePriority(Td * ptd, int newPri)
 */
KlO
EmTpChangePriorityKl(KlO winId, KlO newPri)
{
    Td *ptd;

    KlMustBeIntOrConstInt(winId, 0);
    ptd = EmWinToThread(KlNumToLong(winId));
    KlMustBeIntOrConstInt(newPri, 1);
    EmTpChangePriority(ptd, KlNumToLong(newPri));
    return newPri;
}

/* Set the tick used by Yield to see if enogh time is ellapsed since
 * last give up to scheduler
 */
static void
EmTpSetTick(void)
{
/*    printf("------------set tick\n");*/
    EmTpTick = 1;
}

/* Restart the tick alarm
 */
void
EmTpRestartTick(void)
{
    struct itimerval tickLenght;

/*    printf("------------restart tick\n");*/
    EmTpTick = 0;
    tickLenght.it_value.tv_sec = 0;
    tickLenght.it_value.tv_usec = EMTP_TICK_LENGHT;
    timerclear(&(tickLenght.it_interval));
    setitimer(ITIMER_REAL, &tickLenght, NULL);
}

/* Init the tick alarm
 */
static void
EmTpInitTimer(void)
{
    signal(SIGALRM, (void (*) (int)) EmTpSetTick);
    EmTpRestartTick();
}


/* Call this before any other thread function
 */
static void
EmTpInitInternal(void)
{
    int priority;

    EmTpInitTimer();
    for (priority = EM_TP_MAX_PRI_QUEUE; priority >= 0; priority--) {
	EmTpRunQA[priority] = lCreate(keyVoid, ordVoid, EmTpTdFree, EmTpTdPrint);
    }
    /* create the global list to store each created thread */
    EmTpThreadList = lCreate(keyVoid, ordVoid, EmTpTdFree, EmTpTdPrint);
}

/* static function not documented...
 */
static void
EmTpOnly(void *pu, void *pt, qt_userf_t * f)
{
    ((TpF *) f) ((Td *) pu);
    EmTpAbort((Td *) pt);
    /* NOTREACHED */
}

/* static function not documented...
 */
static void
EmTpStartHelp(qt_t * old, void *ignore0, void *ignore1)
{
    EmTpMainThread.sp = old;
}


/* Create a new thread.
 *
 * Each thread will be linked to a window-formula-group.
 * and include a Td (Thread Data) struct
 */
Td *
EmTpCreateNewThread(TpF * rf /* refresh function */ )
{
    Td *ptd;
    void *sto;

    ptd = UCalloc(1, sizeof(Td));
    /* build a Klone Object to acces this struct with Klone */
    KlIncRef(ptd->self = (KlO) KlStructMake(EmTdClass, ptd));
    ptd->sto = UMalloc(EM_TP_STKSIZE);
    ptd->stackSize = EM_TP_STKSIZE;
    sto = EM_TP_STKALIGN(ptd->sto, QT_STKALIGN);
    ptd->sp = QT_SP(sto, EM_TP_STKSIZE - QT_STKALIGN);
    ptd->sp = QT_ARGS(ptd->sp, (void *) ptd, ptd, (qt_userf_t *) rf, EmTpOnly);
    ptd->status = EM_TP_RUNING;
    ptd->priority = EM_TP_STARTPRIORITY;

    /* put this thread in the runing queue */
    lAdd(EmTpRunQA[ptd->priority], ptd);
    /* add this thread to the global list of threads */
    lAdd(EmTpThreadList, ptd);
    /* start with a 'need refresh' and no 'restart' (not yet started!) */
    EmTpRestartRefresh(ptd, False);
    EmTpNeedRefresh(ptd, True);
    return ptd;
}

/* Check for thread stack overflow
 * print warning on stderr if more than 80% of stack are filled
 */
void
EmTpCheckStackOverflow(Td * self)
{
    double free;

    free = ((((char *) &self) - ((char *) self->sto)) * 100) / (double) (self->stackSize);
/*    fprintf (stderr, "stack free (%%%3.1f) \n", free);*/
    if (free < 20.0) {
	fprintf(stderr,
	"\n\n***** WARNING *****\n few stack free (%%%3.1f) in thread %s \n",
		free,
	    KlStringToCharPtr(EmAGetC(self->pg, EmGateNode, "*group-name*"))
	    );
    }

}

/* static function not documented...
 */
static void
EmTpYieldHelp(qt_t * sp, void *old, void *null)
{
    ((Td *) old)->sp = sp;
}


/* The current thread stops running but stays runable.
 * It is an error to call `EmTpYield' before `EmTpSchedule*'
 * is called or after `EmTpStart' returns.
 */
void
EmTpYield(Td * self)
{
    int isMsg;

#ifdef TEST_THREAD_OVERFLOW
    EmTpCheckStackOverflow(self);
#endif
    EmWaitForNewEvents(False, &isMsg);
    if ((isMsg) || (!self->needRefresh)) {
	QT_BLOCK((qt_helper_t *) EmTpYieldHelp, self, (void *) NULL, EmTpMainThread.sp);
    }
    if (self->status == EM_TP_DYING) {
	EmTpAbort(self);
    }
    if (self->restartRefresh) {
	/* ?? free structs */
	longjmp(self->restartPoint, 1);
    }
}

/* static function not documented...
 */
static void
EmTpAbortHelp(qt_t * sp, void *old, void *null)
{
    Td *ptd;

    ptd = (Td *) old;
    /* unlink this thread Td from the global list */
    lDel(EmTpThreadList, ptd, False);
}

/* Like `EmTpYield' but the thread is discarded. Any intermediate
 * state is lost. The thread can also terminate by simply
 * returning.
 */
void
EmTpAbort(Td * old)
{
    QT_ABORT((qt_helper_t *) EmTpAbortHelp, old, (void *) NULL, EmTpMainThread.sp);
}


/* When one or more threads are created by the main thread,
 * the system goes multithread when this is called.
 * the threads are runned until refresh is done
 * or some events occurs (needs to be handled)
 * they are run in priority order
 */
void
EmTpScheduleHighest(void)
{
    int priority, t, nbe, isMsg, stat;
    Td *tToRun = NULL, *tToCheck = NULL;
    Bools sweep;

    if (!lNbElts(EmTpThreadList))	/* no thread */
	return;
    /* find highest priority thread */
    sweep = False;
    priority = EM_TP_MAX_PRI_QUEUE;
    while (priority > 0) {
	nbe = lNbElts(EmTpRunQA[priority]);
	for (t = 0; t < nbe; t++) {	/* check all thread of this priority */
	    tToCheck = lLookNth(EmTpRunQA[priority], t);
	    if ((tToCheck->status == EM_TP_RUNING) && (tToCheck->needRefresh)) {
		tToRun = tToCheck;
		/* schedule this thread until finished or events */
		/* while no X event and no external message */
		while (tToRun->needRefresh) {
		    if ((stat = EmWaitForNewEvents(False, &isMsg)))
			EmCommErrorKl(stat);
		    if (isMsg)		/* there's a msg */
			return;
		    else		/* call this thread */
			QT_BLOCK((qt_helper_t *) EmTpStartHelp, 0, 0, tToRun->sp);
		}
	    }
	    else if (tToCheck->status == EM_TP_DYING) {
		sweep = True;
		lMarkToSweep(EmTpRunQA[priority], tToCheck);
		/*
		 * this call to QT_BLOCK return in the EmTpYield of the
		 * called thread, where an ABORT is done 
		 */
		QT_BLOCK((qt_helper_t *) EmTpStartHelp, 0, 0, tToCheck->sp);
	    }
	}
	priority--;
    }
    if (sweep) {
	int leftThreads = 0;

	/* some threads are dying... sweep them to death ;) */
	for (priority = EM_TP_MAX_PRI_QUEUE; priority > 0; priority--) {
	    leftThreads += lSweep(EmTpRunQA[priority], True);
	}
    }
}

/* Wrapper:
 */
KlO
EmTpScheduleHighestKl(void)
{
    EmTpScheduleHighest();
    return NIL;
}


/* Wait until a new message arrives on any connection line or on the X server
 * line or a new client ask for a connection line
 */
int
EmWaitForNewEvents(Bools blocking /* must block or not (True or False) */ ,
		   int *pIsMsg		/* if not NULL return if there's a
		       message (X or line) */ )
{
    static fd_set inputFds;
    static int lastUSMustRebuildFDSet = -1;
    static Display *lastEmDisplay = NULL;
    static int maxSock, xfd, stat;
    static struct timeval timeout;
    fd_set readfds;


    if (EmDisplay && XEventsQueued(EmDisplay, QueuedAlready)) {
	if (pIsMsg)
	    *pIsMsg = 1;
	return 0;
    }
    if ((USMustRebuildFDSet != lastUSMustRebuildFDSet) || (EmDisplay != lastEmDisplay)) {
	lastUSMustRebuildFDSet = USMustRebuildFDSet;
	lastEmDisplay = EmDisplay;
	FD_ZERO(&inputFds);
	maxSock = USFillFdAllLines(&inputFds);
	if (EmDisplay) {
	    xfd = XConnectionNumber(EmDisplay);
	    FD_SET(xfd, &inputFds);
	    maxSock = Max(maxSock, xfd);
	}
	(timeout).tv_sec = 0;
	(timeout).tv_usec = 0;
    }
    readfds = inputFds;
    if (blocking) {
	/* block until event */
	if (URestartSelect(maxSock + 1,
			   (fd_set *) & readfds,
			   (fd_set *) NULL,
			   (fd_set *) NULL,
			   (struct timeval *) NULL) < 0) {
	    URetErrno;
	}
	if (pIsMsg)
	    *pIsMsg = 1;
    }
    else {
	/* just check if there's something waiting on these fd */
	if ((stat = URestartSelect(maxSock + 1,
				   (fd_set *) & readfds,
				   (fd_set *) NULL,
				   (fd_set *) NULL,
				   &timeout)) < 0) {
	    URetErrno;
	}
	if (pIsMsg)
	    *pIsMsg = stat;
    }
    return 0;
}

/* Wrapper:
 */
KlO
EmWaitForNewEventsKl(void)
{
    EmCheckError(EmWaitForNewEvents(True, (int *) NULL));
    return NIL;
}

/* Make klone class for thread data (Td)
 */
static void
EmTdClassMake(void)
{
/* warning this number must be the number of declared slots */
#define TdSlotsNumber 2

#define EmDeclareTdField(type,  name, field) \
    KlDeclareStructClassSlot(EmTdClass, name, \
			     KlStructAccessorScalar[sizeof(type)], \
			     KlOffsetOf(Td,field), 0)

    KlIncRef(EmTdClass = KlStructClassMake("Td", sizeof(Td), TdSlotsNumber));
    EmDeclareTdField(Display *, "dpy", dpy);
    EmDeclareTdField(Window, "win", win);
    EmDeclareTdField(Bools, "restartRefresh", restartRefresh);
    EmDeclareTdField(Bools, "needRefresh", needRefresh);

#undef TdSlotsNumber
#undef EmDeclareTdField

}

/* Define some constants used from Klone... priorities
 */
static void
EmDefTpConstant(void)
{
    KlConstantMake("etp:low-priority", KlNumberMake(EM_TP_NORMALPRIORITY));
    KlConstantMake("etp:high-priority", KlNumberMake(EM_TP_FOCUSPRIORITY));
}



/* Initialize the EmTp module
 *
 * Create the Td classe.
 * Define all the Tp subroutines
 */
void
EmTpInit(void)
{
    EmTpInitInternal();
    EmTdClassMake();
    EmDefTpConstant();
    KlDeclareSubr(EmTpRestartRefreshKl, "etp:restart-refresh", 1);
    KlDeclareSubr(EmTpNeedRefreshKl, "etp:need-refresh", 1);
    KlDeclareSubr(EmTpChangePriorityKl, "etp:change-priority", 2);
    KlDeclareSubr(EmTpScheduleHighestKl, "etp:schedule-highest-priority", 0);
    KlDeclareSubr(EmWaitForNewEventsKl, "etp:wait-for-new-events", 0);
}

/* A raw draw function
 * ?? TODO: dbl buffering
 */
void
EmDrawWindow(Td * ptd)
{
    GBox *pgb;
    int ts = 1;
    EmGroup *pg;
    EmNode *pn;
    KlO tmp;

    while (1) {
	EmTpRestartPoint(ptd);
	EmTpRestartRefresh(ptd, False);
	EmTpNeedRefresh(ptd, True);


	pg = ptd->pg;
	pn = KlNumToPtr(EmAGetC(pg, EmGateNode, "*root-node*"));
	ptd->bgc = EmFontToGC(pg, ptd,
			      EmFindFont(pg, pn, "body", 0),
			(Pixel *) KlNumToPtr(EmAGetC(pg, pn, "background")),
		       (Pixel *) KlNumToPtr(EmAGetC(pg, pn, "background")));
	ptd->fgc = EmFontToGC(pg, ptd,
			      EmFindFont(pg, pn, "body", 0),
			(Pixel *) KlNumToPtr(EmAGetC(pg, pn, "foreground")),
		       (Pixel *) KlNumToPtr(EmAGetC(pg, pn, "foreground")));
	XFillRectangle(ptd->dpy, ptd->win, ptd->bgc, 0, 0, ptd->ww, ptd->wh);

	pgb = NULL;
/*	tmp = EmAGetC(pg, NULL, "*gbox-root*");
	if ((tmp) && (tmp != NIL))
	    pgb = KlNumToPtr(tmp);
	else {*/
	pgb = EmBuildGBox(ptd->pg, pn);
	pgb->pFather = NULL;
	EmASetC(pg, EmGateNode, "*gbox-root*", (KlO) KlNumberMake(pgb));
/*	}*/
	EmFillGBox(pg, pgb, GBCNone, ptd->wx, ptd->wy, ptd->ww, ptd->wh, 0, 0, 0);

	pgb->x = pgb->y = 0;
	EmDrawGBox(pg, pgb, ptd, ptd->wx, ptd->wy, ptd->ww, ptd->wh, 0, 0, 0);
	XFlush(ptd->dpy);

	EmTpNeedRefresh(ptd, False);
	EmTpYield(ptd);
    }
}
