#include <X11/Xlib.h>
#include "emselect.h"

#define MAX_TEXT_LEN	500

typedef struct InternSelection {
    Atom xa_PRIMARY;
    Atom xa_TEXT;
    Atom xa_STRING;
    Atom xa_TARGETS;
    Atom xa_ATOM;
    Atom xa_LENGTH;
    Atom xa_INTEGER;
    char *theSelection;
}

                InternSelection;

Boolean 
selectionRequestCB(Widget w, Atom * selection, Atom * target, Atom * type, XtPointer * value, unsigned long *length, int *format);
void 
lostOwnershipCB(Widget w, Atom * selection);
void 
targetCB(Widget w, XtPointer client_data, Atom * selection, Atom * type, XtPointer value, unsigned long *length, int *format);
void 
valueCB(Widget w, XtPointer client_data, Atom * selection, Atom * type, XtPointer value, unsigned long *length, int *format);
Atom 
pick_best_target(Atom * target_list, unsigned long n, InternSelection * is);
void 
my_error_dialog(char *str);


/* Initialize the selection mecanism.
 */
void
EmInitSelection(Display * dpy)
{
    InternSelection *is;		/* Intern Selection Struct */

    is = (InternSelection *) malloc(sizeof(InternSelection));

    is->xa_PRIMARY = XInternAtom(dpy, "PRIMARY", False);
    is->xa_STRING = XInternAtom(dpy, "STRING", False);
    is->xa_TEXT = XInternAtom(dpy, "TEXT", False);
    is->xa_TARGETS = XInternAtom(dpy, "TARGETS", False);
    is->xa_LENGTH = XInternAtom(dpy, "LENGTH", False);
    is->xa_ATOM = XInternAtom(dpy, "ATOM", False);
    is->xa_INTEGER = XInternAtom(dpy, "INTEGER", False);

    EmASetC(EmDisplayToGroup(dpy), EmGateNode, "*selection*", KlNumberMakeIncRef((long) is));
    EmASetC(EmDisplayToGroup(dpy), EmGateNode, "*own-selection*", KlNumberMakeIncRef(0));
}

/* Make Selected formula available from other applications
 * via the cut and paste in X environment.
 */
void
EmOwnSelection(Widget w, Time timestamp, char *msg)
{
    InternSelection *is;		/* Intern Selection Struct */
    int own_primary;
    Display *dpy;

    dpy = XtDisplay(w);
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), EmGateNode, "*selection*"));

    own_primary = KlNumToInt(EmAGetC(EmDisplayToGroup(dpy), EmGateNode, "*own-selection*"));

    /*
     * Disown selection if we owned one 
     */
    if (own_primary) {
	XtDisownSelection(w, is->xa_PRIMARY, timestamp);
	own_primary = False;
    }

    /*
     * Let's tell we have something selected 
     */
    if (msg != NULL) {
	is->theSelection = (char *) malloc(sizeof(char) * (strlen(msg) + 1));

	strcpy(is->theSelection, msg);

	own_primary = XtOwnSelection(w, is->xa_PRIMARY, timestamp,
				     selectionRequestCB,
				     lostOwnershipCB,
				     NULL);
	/*
	 * make sure we have ownership if not, clear selection data 
	 */
	if (!own_primary)
	    lostOwnershipCB(w, &(is->xa_PRIMARY));
    }

    EmASetC(EmDisplayToGroup(dpy), NULL, "*own-selection*", (KlO) KlNumberMake(own_primary));
/*    printf("Salut:%s\n", is->theSelection);*/
}

/* Wrapper: for EmOwnSelection(Widget w, Time timestamp, char *msg)
 */
KlO
EmOwnSelectionKl(KlO g, KlO timestamp, KlO msg)
{
    Time truc;
    Display *dpy;
    InternSelection *is;
    Widget wid;

    KlMustBeIntOrConstInt(g, 0);
    KlMustBeNumber(timestamp, 1);
    KlMustBeString(msg, 2);

    fprintf(stderr, "Own Selection\n");

    dpy = KlNumToPtr(EmAGetC(KlNumToGrp(g), EmGateNode, "*display*"));
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), EmGateNode, "*selection*"));

    truc = KlNumToLong(timestamp);
    wid = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), EmGateNode, "*selection-widget*"));

    EmOwnSelection(wid, truc, KlStringToCharPtr(msg));
    return msg;
}


/******************************************************
  
    Stuff for obtaining selection value
  
*******************************************************/

/* Someone has asked us for the value of a selection.
 *
 * Respond appropriately, based on the target they specified.
 */
Boolean
selectionRequestCB(Widget w /* In */ ,
		   Atom * selection /* In */ ,
		   Atom * target /* In */ ,
		   Atom * type /* Out */ ,
		   XtPointer * value /* Out */ ,
		   unsigned long *length /* Out */ ,
		   int *format /* Out */ )
{
    InternSelection *is;		/* Internc Selection Struct */
    int i;
    XSelectionRequestEvent *req;
    Atom *tmp_target_list;
    int *tmp_length;
    Display *dpy;

    printf("Selection Request\n");
    dpy = XtDisplay(w);
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), EmGateNode, "*selection*"));

    req = XtGetSelectionRequest(w, *selection, NULL);

    /*
     * First, handle requests for TARGETS This is when you're asked for
     * targets 
     */
    if (*target == is->xa_TARGETS) {
	Atom *xmu_targets;
	unsigned long xmu_length;

	/*
	 * Get Xmu to list targets it handles 
	 */
	XmuConvertStandardSelection(w, req->time, selection, target, type,
				    &xmu_targets, &xmu_length, format);
	/*
	 * Put our targets in the list. 
	 */
	*length = xmu_length + 3;	/* 3 is number of targets we support */
	tmp_target_list = (Atom *) XtMalloc(sizeof(Atom) * (*length));
	tmp_target_list[0] = is->xa_STRING;
	tmp_target_list[1] = is->xa_TEXT;
	tmp_target_list[2] = is->xa_LENGTH;
	/*
	 * Copy the Xmu targets into our list 
	 */
	for (i = 3; i < *length; i++)
	    tmp_target_list[i] = xmu_targets[i - 3];
	XtFree((char *) xmu_targets);
	*value = (XtPointer) tmp_target_list;
	*type = is->xa_ATOM;
	*format = 32;
	return True;
    }

    /*
     * Handle a request for STRING or TEXT Malloc space and copy the string
     * in. Just pass a pointer to the appropriate part of the string. The
     * string is: is->theSelection 
     */
    if ((*target == is->xa_TEXT) || (*target == is->xa_STRING)) {
	*length = strlen(is->theSelection);
	*value = XtMalloc(*length + 1);
	strcpy(*(char **) value, is->theSelection);
	*type = is->xa_STRING;
	*format = 8;
	return True;
    }

    /*
     * Handle a request for LENGTH 
     */
    if ((*target == is->xa_LENGTH)) {
	tmp_length = (int *) XtMalloc(sizeof(int));

	*tmp_length = strlen(is->theSelection) + 1;
	*value = (XtPointer) tmp_length;
	*length = 1;
	*type = is->xa_INTEGER;
	*format = sizeof(int) * 8;

	return True;
    }

    /*
     * We couldn't do it, maybe Xmu can 
     */
    return (XmuConvertStandardSelection(w, req->time, selection, target,
					type, value, length, format));
}


/* This is the callback when you lose ownership
 */
void
lostOwnershipCB(Widget w, Atom * selection)
{
    InternSelection *is;		/* Intern Selection Struct */
    Display *dpy;

    printf("Lost Ownership\n");
    dpy = XtDisplay(w);
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), EmGateNode, "*selection*"));

    /*
     * If we lose ownership of primary, clean up our highlight and reset our
     * state. 
     */
    if (*selection == is->xa_PRIMARY) {
	EmASetC(EmDisplayToGroup(dpy), NULL, "*own-selection*", (KlO) KlNumberMake(0));
	/*
	 * Here, you do the stuff about cleaning up highlight. The highlight
	 * attribute must be turned off 
	 */
	NULL;
    }
}


/*
 * User asked to get PRIMARY selection.
 * First find out what formats are available.
 */
void
EmGetPrimary(Widget w, Time timestamp)
{
    InternSelection *is;		/* Intern Selection Struct */
    Display *dpy;

    dpy = XtDisplay(w);
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), NULL, "*selection*"));

    XtGetSelectionValue(w, is->xa_PRIMARY, is->xa_TARGETS, targetCB,
			(XtPointer) timestamp, timestamp);
}

/* Wrapper: for EmOwnSelection(Widget w, Time timestamp, char *msg)
 */
KlO
EmGetPrimaryKl(KlO g, KlO timestamp)
{
    Time truc;
    Widget wid;

    KlMustBeIntOrConstInt(g, 0);
    KlMustBeNumber(timestamp, 1);

    truc = KlNumToLong(timestamp);
    wid = KlNumToPtr(EmAGetC(KlNumToGrp(g), EmGateNode, "*selection-widget*"));

    fprintf(stderr, "Get Primary\n");
    EmGetPrimary(wid, truc);
    return NIL;
}

/*
 * Pick the best format from the list available and ask
 * for a conversion to that target.
 */
void
targetCB(Widget w,
	 XtPointer client_data,
	 Atom * selection,
	 Atom * type,
	 XtPointer value,
	 unsigned long *length,
	 int *format)
{
    Atom best_target;
    Time paste_time = (Time) client_data;
    InternSelection *is;		/* Intern Selection Struct */
    Display *dpy;

    dpy = XtDisplay(w);
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), NULL, "*selection*"));

    if (*type == XT_CONVERT_FAIL) {
	my_error_dialog("Paste operation timed out, try again.");
	return;
    }

    if ((*type == None) || (*length == 0)) {
	/* Could be that owner doesn't support TARGETS, just try STRING */
	best_target = is->xa_STRING;
    }
    else {
	best_target = pick_best_target((Atom *) value, *length, is);
    }

    if (!best_target)
	my_error_dialog("No appropriate data to paste");
    else
	XtGetSelectionValue(w, *selection, best_target, valueCB,
			    NULL, paste_time);
}


/*
 * We got the data, let's tack it on to our string.
 */
void
valueCB(Widget w,
	XtPointer client_data,
	Atom * selection,
	Atom * type,
	XtPointer value,
	unsigned long *length,
	int *format)
{
    int current_length;
    int chars_to_copy, i;
    char *tmp_ptr;
    InternSelection *is;		/* Intern Selection Struct */
    Display *dpy;

    dpy = XtDisplay(w);
    is = KlNumToPtr(EmAGetC(EmDisplayToGroup(dpy), NULL, "*selection*"));

    fprintf(stderr, "We've got Datas !!!\n");
    /*
     * Check various possibilities for error. 
     */

    if (*type == XT_CONVERT_FAIL) {
	my_error_dialog("Paste operation timed out, try again.");
	return;
    }
    if (*type == None) {
	my_error_dialog("No selection, or Paste operation failed.");
	return;
    }
    if ((*type != is->xa_STRING) && (*type == is->xa_TEXT)) {
	XtFree((char *) value);
	my_error_dialog("Attempt to paste unsupported data");
	return;
    }

    fprintf(stderr, "Coller: %s\n", value);

    XtFree((char *) value);
}


/**************************************************************
  
		       Utilities
  
**************************************************************/

/* Go through our list of preferred targets and see if
 * they are supported by the owning client. Return the
 * first match.
 *
 * If none match, return NULL.
 */
Atom
pick_best_target(Atom * target_list, unsigned long n, InternSelection * is)
{
    int i, j;
    Atom *preferred_targets[] =
    {
     &(is->xa_STRING),
     &(is->xa_TEXT)
    };


    for (i = 0; i < XtNumber(preferred_targets); i++)
	for (j = 0; j < n; j++)
	    if (target_list[j] == *preferred_targets[i])
		return (target_list[j]);

    return (Atom) NULL;
}


/*
 * Put up a dialog with the given message.
 */
void
my_error_dialog(char *str)
{
    fprintf(stderr, "Error Cut/Paste : %s\n", str);
}


/* Init converters
 */
void
EmSelectionInit(void)
{
    KlDeclareSubr(EmOwnSelectionKl, "own-selection", 3);
    KlDeclareSubr(EmGetPrimaryKl, "get-primary", 2);
}
